#ifndef ROBOT_H
#define ROBOT_H

#include "mbed.h"
#include "led.h"
#include "sensors.h"
#include "motors.h"
#include "calibration.h"
#include "serialcomms.h"

#define LED_ADDRESS 0xC0
#define ADC_ADDRESS 0x90
#define COMMAND_MESSAGE_BYTE 0x1D
#define ACKNOWLEDGE_MESSAGE_BYTE 0x1E
#define RESPONSE_MESSAGE_BYTE 0x1F
#define STATUS_MESSAGE_BYTE 0x1C
#define IR_MESSAGE_BYTE 0x1B
#define SOFTWARE_VERSION_CODE 0.10

// SERIAL INTERFACES SETTINGS 

/* ENABLE_BLUETOOTH [1=on, 0=off]:  Enable if the BlueSmirf module is being used */
/** @brief Enable if the BlueSmirf module is being used. 0=off 1=on*/
#define ENABLE_BLUETOOTH 1

/* ENABLE_PC_SERIAL [1=on, 0=off]:  Enable if the PC(RPi) USB serial module is being used */
#define ENABLE_PC_SERIAL 1

/* BLUETOOTH_BAUD [recommended=115200]:  Baud rate for the BlueSMIRF module */
#define BLUETOOTH_BAUD 115200

/* PC_BAUD [recommended=460800 for optimal performance with RPi, 115200 for compatability]:  Baud rate for the RPi USB serial module */
//#define PC_BAUD 115200
#define PC_BAUD 460800

/* DEBUG_MODE [1=on, 0=off]:  Enable to allow debug messages to be sent of one of the serial interfaces */
#define DEBUG_MODE 1

/* DEBUG_OUTPUT_STREAM [1=PC\USB 2=BlueSmirf 4=Display]:  Specify which output stream(s) should be used by default for debug messages, if enabled.  Recommended to use BT for debug messages.*/
#define DEBUG_OUTPUT_STREAM 2

// To update sensors 10 times a second (8 x 0.0125 = 0.1)
#define SENSOR_TICKER_PERIOD 0.0125

// H-Bridge should work at upto 100kHz (10uS) but note it seems to behave unusually at frequencies close to but above this
// Slower speeds work a bit faster but noisier
#define MOTOR_PWM_PERIOD_US 10

#define USE_STALL_OFFSET 1
#define STALL_OFFSET 0.22

extern I2C primary_i2c;
extern DigitalIn rpi1;
extern DigitalOut case_led;
extern DigitalOut mbed_led1;
extern DigitalOut mbed_led2;
extern DigitalOut mbed_led3;
extern DigitalOut mbed_led4;
extern DigitalOut case_led;
extern AnalogIn vin_battery;
extern Serial pc;
extern Serial bt;
extern SerialComms serial;
extern Led led;
extern Sensors sensors;
extern Motors motors;
extern volatile char i2c_lock;
extern char debug_mode;
extern char debug_output;
extern char status_message [];

/**
 *  The Robot class contains the core functions for the robot
 */
class Robot
{
public:

    /**
     * Main initialisation routine: setup the robot, the I2C interfaces, start system timers etc.
     *
     */
    void init(void);
    
    /**
     * Get the uptime for the MBED
     *
     * @return The amount of time in seconds that the MBED has been active since last reset
     */
    float get_uptime(void);
    
    /**
     * Get the battery voltage
     * 
     * The battery voltage is passed through a 7.5V Zener diode, then into a 1:1 potential divider.   This
     * allows a voltage in the approximate range 7.5 to 13.8V to be measured.
     *
     * @return The current voltage reading for the battery
     */
    float get_battery_voltage(void);
    
    /**
     * Display a string message (printf) on the selected debug output stream [eg pc, bt or both]
     */
    void debug(const char* format, ...);
    
    
    /**
     * Toggles the current state of the MBED case led (use with a ticker to blink LED etc)
     */
    void case_led_toggle(void);
    
    
    /**
     * Update the global 8-character status message
     */ 
    void update_status_message(void);
    
    private:
    void _update_minutes(void);
};


extern Robot robot;
#endif