#include "Timer2.h"

Timer2::Timer2(int period, timerCb_t p_callback) {
    this->callback = p_callback;
    
    LPC_SC->PCONP |= (1 << 22);         // Timer2 power on
    LPC_SC->PCLKSEL1 |= (1 << 12);      // Divider of 1 for the CLK, Set PCLK_TIMER2 to CCLK
    
    LPC_PINCON->PINSEL0 |= (3 << 12);   // P0.6 = MAT2.0
    LPC_PINCON->PINSEL0 |= (3UL << 8);  // P0.4 set to function CAP2.0 on p30
    LPC_PINCON->PINSEL0 |= (3UL << 10); // P0.5 set to function CAP2.1 on p29

    LPC_TIM2->CCR |= (1UL);      // Capture CAP2.0 rising edge. CR0 will be loaded with content of TC
    LPC_TIM2->CCR |= (1UL << 2); // Interrupt on CAP2.0 event. 
    LPC_TIM2->CCR |= (1UL << 4); // Capture CAP2.1 falling edge. CR1 will be loaded with the contents of TC.
    //LPC_TIM2->CCR |= (1LU << 5); // Interrupt on CAP2.1 event.     
    
    LPC_TIM2->IR |= 0xFFFFFFFF;         // clear Timer2 interrupt register
    
    LPC_TIM2->MCR |= (1 << 0);          // Set the timer to generate interupt when TC reach MR0
    LPC_TIM2->MR0 = period;
    LPC_TIM2->EMR &= ~(1 << 0);         // set the first output to be low
    
    this->setEMC(TIMER2_EMC_TOGGLE);
    
    LPC_TIM2->TCR |= (1 << 1);          // reset timer
    LPC_TIM2->TCR &= ~(1 << 1);         // release reset
}

Timer2::~Timer2() {
    LPC_TIM2->TCR &= ~(1 << 0);          // Stop Timer2
    LPC_SC->PCONP &= ~(1 << 22);         // Timer2 power off
    NVIC_DisableIRQ(TIMER2_IRQn);
}

void Timer2::Set_RE(bool b)
{
    if(b)
    {
        LPC_TIM2->CCR |= (1UL << 2); // Activate Interrupt on CAP2.0 event. 
    }
    else
    {
        LPC_TIM2->CCR &= ~(1UL << 2); // Deactivate Interrupt on CAP2.0 event. 
    }
}

void Timer2::Set_FE(bool b)
{
    if(b)
    {
        LPC_TIM2->CCR |= (1LU << 5); // Activate Interrupt on CAP2.1 event.
    }
    else
    {
        LPC_TIM2->CCR &= ~(1LU << 5); // Deactivate Interrupt on CAP2.1 event.
    }
}

void Timer2::start(void) {
    NVIC_SetVector(TIMER2_IRQn, (uint32_t) callback);
    NVIC_EnableIRQ(TIMER2_IRQn);
    NVIC_SetPriority(TIMER2_IRQn,0);
    
    LPC_TIM2->TCR |= (1 << 0);          // start Timer2
}

void Timer2::setMatchRegister(unsigned int value) {
    LPC_TIM2->MR0 = value;
}

void Timer2::incrementMatchRegister(int value) {
    LPC_TIM2->MR0 += value;
}

bool Timer2::getIntAtPos(int pos) {
    if(LPC_TIM2->IR & (1 << pos))
        return true;
    else
        return false;
}

void Timer2::clearIntAtPos(int pos) {
    LPC_TIM2->IR |= (1 << pos);
}

void Timer2::setEMC(int value) {
    switch(value) {
        case TIMER2_EMC_NOT:
            LPC_TIM2->EMR &= ~(3 << 4);
            break;
        case TIMER2_EMC_LOW:
            LPC_TIM2->EMR |= (1 << 4);
            LPC_TIM2->EMR &= ~(1 << 5);
            break;
        case TIMER2_EMC_HIGH:
            LPC_TIM2->EMR |= (1 << 5);
            LPC_TIM2->EMR &= ~(1 << 4);
            break;
        case TIMER2_EMC_TOGGLE:
            LPC_TIM2->EMR |= (3 << 4);
            break;
     }
}