/*!
 *  \file main.cpp
 *  \brief Main file and initialize interrupt function
 *  \author Equipe P02
 *  \version 1
 *  \date 02/04/2014
 */
 
#include "accelerometer.h"
#include "analyzer.h"
#include "interrupt.h"
#include <iostream>
#include <fstream>
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

DigitalOut led1(LED1);  // Debug led
DigitalOut led2(LED2);  // Debug led
DigitalOut led4(LED4);  // Debug led
DigitalIn button(p30);  // Push button

LocalFileSystem local("local"); // Open config file

void *accelerometer = Accelerometer_C_new();    // Accelerometer object in C
void *analyzer = Analyzer_C_new();              // Analyzer object in C

bool rebound = true;    // For rebound
bool appuyer = false;   // For push bouton

extern "C" void interruptCapture(void)
{
    if(rebound && !appuyer)
    {
        LPC_TIM2->TC = 0;       // clear timer counter
        LPC_TIM2->PC = 0;       // clear prescale counter
        LPC_TIM2->PR = 0;       // clear prescale register
        LPC_TIM2->MCR |= 0x03;  // interrupt and reset control
        LPC_TIM2->CCR = 0;  //no capture interupt
        rebound = false;
    }
    else if(!rebound && !appuyer)
    {
        led1 = 1;
        LPC_TIM2->MCR = 0;
        LPC_TIM2->CCR = (0x06 << 0);  //falling edge
        
        LPC_TIM1->TC = 0;       // clear timer counter
        LPC_TIM1->PC = 0;       // clear prescale counter
        LPC_TIM1->PR = 0;       // clear prescale register
        LPC_TIM1->IR |= 0xFF;
        LPC_TIM1->TCR = 0x01;   //enable timer 3
    
        appuyer = true;
        NVIC_EnableIRQ(TIMER1_IRQn); // Enable TIMER1 interrupt
        
        signed char* values = Accelerometer_C_getAccelValue(accelerometer);
        
        // Analyze data
        Analyzer_C_setInitial(values, analyzer);
    }
    else
    {
        led1 = 0;
        LPC_TIM2->CCR = (5 << 0);  //rising edge
        
        LPC_TIM1->TC = 0;       // clear timer counter
        LPC_TIM1->PC = 0;       // clear prescale counter
        LPC_TIM1->PR = 0;       // clear prescale register
        LPC_TIM1->IR |= 0xFF;
        LPC_TIM1->TCR = 0;      //disable timer 3
        
        appuyer = false;
        rebound = true;
        NVIC_DisableIRQ(TIMER1_IRQn); // Disable TIMER1 interrupt
        
        Analyzer_C_checkMouvement(analyzer);    // Find mouvement
    }
    
    LPC_TIM2->IR |= 0xFF;   // Reset interrupt
}

extern "C" void interruptMatch(void)
{
    if(button)
    {
        led4 = !led4;   // Debug led
        
        signed char* values = Accelerometer_C_getAccelValue(accelerometer); // Get value of accelerometer
        
        // Analyze data
        Analyzer_C_setMinMax(values, analyzer);
        
        delete values;
    }
    
    LPC_TIM1->IR |= 0xFF; // Reset interrupt
}

void initialize()
{
    //Initialize the analyzer
    Analyzer_C_Initialize(analyzer);

    //  CAPTURE 
    //  Enable the pins on the device to use TIMER CAP2.0 on pin 30
    LPC_PINCON->PINSEL0 |= 3 << 8;     //set pin 30 capture 2.0
    LPC_SC->PCONP |= (1 << 22);        // Timer2 power on
    
      //LPC_TIM2->MCR |= 3;             //interrupt and reset control
    LPC_TIM2->TC = 0;                   // clear timer counter
    LPC_TIM2->PC = 0;                   // clear prescale counter
    LPC_TIM2->PR = 0;                   // clear prescale register
    LPC_TIM2->MR0 = SystemCoreClock / 100; //match pour rebound
    
    LPC_TIM2->IR |= 0xFF;               // Clear CR0 interrupt flag
    LPC_TIM2->CCR = (5 << 0);           // enable cap2.0 rising-edge capture; interrupt on cap2.0 p.496
    LPC_TIM2->TCR = (1 << 1);           // reset timer
    LPC_TIM2->TCR = 1;                  // enable timer
    
    NVIC_EnableIRQ(TIMER2_IRQn);        // Enable timer2 interrupt
    NVIC_SetVector(TIMER2_IRQn, uint32_t(interruptCapture));
    
    //  TIMER 3
    //LPC_SC->PCONP |= 1 << 23;           // TIMER3 power on
        
    LPC_TIM1->MCR = 3;                  // Interrupt and reset control
    LPC_TIM1->TC = 0;                   // clear timer counter
    LPC_TIM1->PC = 0;                   // clear prescale counter
    LPC_TIM1->PR = 0;                   // clear prescale register
    LPC_TIM1->MR0 = SystemCoreClock / 100;
        
    LPC_TIM1->IR |= 0xFF;               // Clear MR0 interrupt flag
    LPC_TIM1->TCR = (1 << 1);           // Reset TIMER1
    LPC_TIM1->TCR = 0;                  // Disable timer
    
    //NVIC_EnableIRQ(TIMER1_IRQn);      // Enable TIMER1 interrupt
    NVIC_SetVector(TIMER1_IRQn, uint32_t(interruptMatch));
}

int main()
{
    string line;
    string filename = "/local/config.txt";
    
    PHY_PowerDown(); //power down ethernet interface 175mw
    
    ifstream myfile(filename.c_str());
    
    if (myfile.is_open())
    {
        while (getline(myfile,line))
        {
            if (line[0] != '#')
            {
                // Find hand
                if (line.find("Hand") != string::npos)
                {
                    if (line.compare(5,6,"L") == 0) // If left hand
                        Analyzer_C_setHand(true, analyzer);
                    else
                        Analyzer_C_setHand(false, analyzer);
                }
            }
        }
        
        myfile.close();
        
        initialize();
        
        while(true)
        {
           Sleep(); // Sleep main thread
        }
    }
    
    return 0;
}
