/*!
 *  \file analyzer.h
 *  \brief Class that analyzes the mouvement in C and C++
 *  \author Equipe P02
 *  \version 0.1
 *  \date 02/04/2014
 */

#ifndef ANALYZER_H
#define ANALIZER_H

#include <mbed.h>
#include <climits>
#include <map>
#include "mouvement.h"

using namespace std;

/*!
 *  \def SEUIL_DETECTION
 *  \brief It's the limit for the detection
 */
#define SEUIL_DETECTION 15

/*! 
 *  \class Analyzer analyzer.h "analyzer.h"
 *  \brief Class that analyzes the mouvement
 *
 *  Class in C++ that analyzes data of acceloremeter and find
 *  the mouvement
 */
class Analyzer
{
public:
    
    /*!
     *  \brief Constructor
     */
    Analyzer();
    
    /*!
     *  \brief Destructor
     */
    ~Analyzer();
    
    /*!
     *  \brief Initialize analyzer
     */
    void initialize();
    
    /*!
     *  \brief Set the minimum and maximum for each axe
     *  \param values Values to compare with the min and max
     */
    void setMinMax(signed char* values);
    
    /*!
     *  \brief Set the initial value for each axe
     *  \param value The initial value
     */
    void setInitial(signed char* value);
    
    /*!
     *  \brief Analyze the mouvement with initial, min and max of each axes
     *
     *  Create value in hexadecimal and find this value in the map
     */
    void checkMouvement();
    
    void setHand(bool l) { hand = l; }
    
private:

    /*! 
     *  \class Data analyzer.h "analyzer.h"
     *  \brief Struct that contains data useful for the analysis
     *
     *  Struct in C++ that contains minimum, maximum, initial value and orientation
     */
    struct Data
    {
        signed char min; /*!< Minimum value of the mouvement */
        signed char max; /*!< Maximum value of the mouvement */
        signed char initial; /*!< Initial value of the mouvement */
        bool sens; /*!< Orientation of the mouvement true =  acceleration, false = deceleration */
        
        /*!
         *  \brief Constructor
         */
        Data(): min(SCHAR_MAX), max(SCHAR_MIN){}
        
        /*!
         *  \brief Set the minimum and maximum
         *  \param value Value to compare with the min and max
         */
        void setMinMax(signed char value)
        {
            if (value < min)
            {
                min = value;
                sens = true;
            }
                
            if (value > max)
            {
                max = value;
                sens = false;
            }
        }
        
        /*!
         *  \brief Set the initial value
         *  \param value The initial value for initial, min and max
         */
        void setInitial(signed char init)
        {
            initial = init;
            min = initial;
            max = initial;
        }

        /*!
         *  \brief Determine is there a mouvement
         *  \return true if detect mouvement, false if not detect mouvement
         */
        bool isMouvement()
        {
            if(sens)
                return (max-initial >= SEUIL_DETECTION);
            
            else
                return (initial-min >= SEUIL_DETECTION);
        }
    };
    
    Data x; /*!< Data for the axe x */
    Data y; /*!< Data for the axe y */
    Data z; /*!< Data for the axe z */
    bool hand; /*!< Char to determine the hand true = left, false = right*/
    
    map<int, Mouvement*> allMouvement; /*!< map that containts all movement */
    Trame *trame; /*!< Object Trame for send information */
};
 
// Define function in C for interruption
// The C function call C++ function
#ifdef __cplusplus
extern "C" {
#endif
 
extern void *Analyzer_C_new(); /*! Call Constructor \see Analyzer() */
extern void Analyzer_C_delete(void *analyzer); /*! Call Constructor \see ~Analyzer() */
extern void Analyzer_C_Initialize(void *analyzer); /*! Call Initialize \see initialize() */
extern void Analyzer_C_setMinMax(signed char* values, void *analyzer); /*! Call setMinMax \see setMinMax(signed char*) */
extern void Analyzer_C_setInitial(signed char* value, void *analyzer); /*! Call setInitial \see setInitial(signed char*) */
extern void Analyzer_C_checkMouvement(void *analyzer); /*! Call checkMouvement \see checkMouvement() */
extern void Analyzer_C_setHand(bool h, void *analyzer); /*! Call setHand \see setHand(bool left) */
 
#ifdef __cplusplus
}
#endif
 
#endif
