#ifndef _pct2075_h_
#define _pct2075_h_

#include "mbed.h"

class PCT2075 {
  private:
    I2C     &i2c;
    uint8_t  address_write;
    uint8_t  address_read;

  public:
    static const int16_t  TEMP_MAX  = 12500; // centi-°C
    static const int16_t  TEMP_MIN  = -5500; // centi-°C
    static const uint16_t TIDLE_MAX =  3100; // ms
    static const uint16_t TIDLE_MIN =   100; // ms

    // Fault queue is defined as the number of faults that must occur
    // consecutively to activate the OS output. It is provided to avoid false
    // tripping due to noise. Because faults are determined at the end of data
    // conversions, fault queue is also defined as the number of consecutive
    // conversions returning a temperature trip.
    enum OSFaultQue {
        OS_FAULT_QUE_1 = 0,
        OS_FAULT_QUE_2 = 1,
        OS_FAULT_QUE_4 = 2,
        OS_FAULT_QUE_6 = 3
    };

    enum OSPolarity {
        OS_ACTIVE_LOW  = 0,
        OS_ACTIVE_HIGH = 1
    };

    enum OSMode {
        OS_MODE_COMP      = 0,
        OS_MODE_INTERRUPT = 1
    };

    // In shutdown mode, the PCT2075 draws a small current of <1.0 µA and the
    // power dissipation is minimized; the temperature conversion stops, but the
    // I2C-bus interface remains active and register write/read operation can be
    // performed. When the shutdown is set, the OS output will be unchanged in
    // comparator mode and reset in interrupt mode.
    enum DeviceMode {
        DEVICE_MODE_NORMAL   = 0,
        DEVICE_MODE_SHUTDOWN = 1
    };

    struct Configuration {
         OSFaultQue os_fault_que;
         OSPolarity os_polarity;
         OSMode     os_mode;
         DeviceMode device_mode;
    };

    PCT2075 (I2C &i2c_obj, uint8_t address = 0x48);

    ~PCT2075();
    int16_t PCT2075_os_temp;
    int16_t PCT2075_hyst_temp;
    uint8_t PCT2075_temp_idle_cycle;

    Configuration get_configuration();
    void set_configuration(Configuration& config);

    // Shorthand functions for changing one parameter of the configuration
    // without changing the other parameters.
    // Note: When changing multiple parameters it is more efficient to batch
    // them using set_configuration.
    void set_os_fault_queue(OSFaultQue fault_que);
    void set_os_polarity(OSPolarity polarity);
    void set_os_mode(OSMode mode);
    void set_device_mode(DeviceMode mode);
    void shutdown_mode();
    void normal_mode();

    // Returns the temperature in centi-degrees C,
    // resolution of .125 degrees, rounded down to .1 degrees
    int16_t read_temperature();

    // Get the OS temperature in centi-degrees C,
    // rounded to .5 degrees C
    int16_t get_os_temperature();
    // Set the OS temperature in centi-degrees C,
    // rounded to .5 degrees C
    void set_os_temperature(int16_t temperature);

    // Get the hysteresis temperature in centi-degrees C,
    // rounded to .5 degrees C
    int16_t get_hyst_temperature();
    // Set the hysteresis temperature in centi-degrees C,
    // rounded to .5 degrees C
    void set_hyst_temperature(int16_t temperature);

    // Get idle time in ms, resolution of 100ms (deci-seconds)
    uint16_t get_idle_time();
    // Set idle time in ms, rounded down to 100ms (deci-seconds)
    // In normal operation mode, the temp-to-digital conversion is executed
    // every 100 ms or other programmed value and the Temp register is updated
    // at the end of each conversion. During each ‘conversion period’ (Tconv) of
    // about 100 ms, the device takes only about 28 ms, called ‘temperature
    // conversion time’ (tconv(T)), to complete a temperature-to-data conversion
    // and then becomes idle for the time remaining in the period. This feature
    // is implemented to significantly reduce the device power dissipation.
    void set_idle_time(uint16_t time);
};

#endif // _PCT2075_H_
