/**
 * @author Thomas Hamain (fait bcp de photes)
 *
 * @section DESCRIPTION
 *  Librairie pour le double pont H L298 prennant simplement en paramètre une valeur de -1 à 1
 *
 */


/**
 * Includes
 */
#include "mbed.h"
#include "L298.h"

/**
 * Constructeur.
 * Initialise la STM32 pour être branchée au L298
 *
 * @param ENA mbed pin pour ENA.
 * @param ENB mbed pin pour ENB.
 * @param ENA mbed pin pour IN1.
 * @param ENB mbed pin pour IN2.
 * @param ENA mbed pin pour IN3.
 * @param ENB mbed pin pour IN4.
 * @param freq_pwm la fréquence de la PWM envoyée au pont H (tester avec valeur
 * negative mais je pense que le compilateur insulte!)
 */
L298::L298(PinName ena,
           PinName enb,
           PinName in1,
           PinName in2,
           PinName in3,
           PinName in4,
           double freq_pwm) : _ena(ena), _enb(enb), _in1(in1), _in2(in2), _in3(in3), _in4(in4), _periode_pwm(1/freq_pwm)
{
    _ena.period(_periode_pwm);
    _enb.period(_periode_pwm);
    _ena.pulsewidth(0);
    _enb.pulsewidth(0);
}

/**
* Constructeur par default
* Initialise par défaut:
* - D6  : ENA (PWM du moteur A)
* - D11 : ENB (PWM du moteur B)
* - D7  : IN1
* - D8  : IN2
* - D9  : IN3
* - D10 : IN4
*
* La fréquence de la PWM envoyée aux moteurs est de 1kHz.
*
* Pour plus d'information sur les pins INx, se référer au fonctionnement du L298 (et completer la doc de la classe)
*/

L298::L298() : _ena(D6), _enb(D11), _in1(D7), _in2(D8), _in3(D9), _in4(D10), _periode_pwm(0.001)
{
    _ena.period(_periode_pwm);
    _enb.period(_periode_pwm);
    _ena.pulsewidth(0);
    _enb.pulsewidth(0);
}

/**
* Contrôle du moteur A.
* Envoie une PWM avec le rapport cyclique allant de -1 à 1. Le signe de la
* valeur passée en paramètre indique la direction. La valeur de zéro
* correspond à un arrêt du moteur. La fréquence de la PWM peut être connue
* ou modifiée grâce à respectivement getPeriodePwm et setPeriodePwm.
*
* @param rc rapport cyclique -1 à 1.
*/
void L298::A(double rc)
{
    if (rc >= 0) {
        _in1 = 1;
        _in2 = 0;
    } else {
        _in1 = 0;
        _in2 = 1;
        rc = -1 * rc;
    }
    _ena.pulsewidth(rc*_periode_pwm);
}

/**
* Contrôle du moteur A.
* Envoie une PWM avec le rapport cyclique allant de -1 à 1. Le signe de la
* valeur passée en paramètre indique la direction. La valeur de 0
* correspond à un arrêt du moteur. La fréquence de la PWM peut être connue
* ou modifiée grâce à respectivement getPeriodePwm et setPeriodePwm.
*
* @param rc rapport cyclique -1 à 1.
*/
void L298::B(double rc)
{
    if (rc >= 0) {
        _in1 = 1;
        _in2 = 0;
    } else {
        _in1 = 0;
        _in2 = 1;
        rc = -1 * rc;
    }
    _enb.pulsewidth(rc*_periode_pwm);
}





/*
PwmOut mypwm_a(D6);
PwmOut mypwm_b(D11);

DigitalOut in1(D7);
DigitalOut in2(D8);
DigitalOut in3(D9);
DigitalOut in4(D10);



void moteur_init()
{
    mypwm_a.period(PERIODE_PWM_MOTEUR);
    mypwm_b.period(PERIODE_PWM_MOTEUR);
    mypwm_a.pulsewidth(0);
    mypwm_b.pulsewidth(0);
}

void moteur_a(float percent)
{

    if (percent>=0) {
        in1 = 1;
        in2 = 0;
    } else {
        in1 = 0;
        in2 = 1;
        percent = -1 * percent;
    }
    mypwm_a.pulsewidth((float)percent*(float)PERIODE_PWM_MOTEUR);
}

void moteur_b(float percent)
{

    if (percent>=0) {
        in4 = 1;
        in3 = 0;
    } else {
        in4 = 0;
        in3 = 1;
        percent = -1 * percent;
    }

    mypwm_b.pulsewidth((float)percent*(float)PERIODE_PWM_MOTEUR);
}*/
