#include "mbed.h"
#include "QEI.h"
#include "Moteur.h"
#include "math.h"

#define PI 3.14159265359

#define DISTANCE_ENTRE_ROUES 197.0 /*en mm*/
#define DIAMETRE_ROUE_A 70.06 /*en mm*/
#define DIAMETRE_ROUE_B 70.02 /*en mm*/
#define TIC_PAR_TOUR_A 1200 /*Nombre de tics de l'encodeur*/
#define TIC_PAR_TOUR_B 1200 /*Nombre de tics de l'encodeur*/
#define DIST_PAR_TIC_A 0.183417 /*Distance parcourue entre 2 tics pour la roue A en mm*/ /*DIAMETRE_ROUE_A*PI/TIC_PAR_TOUR_A*/
#define DIST_PAR_TIC_B 0.183312 /*Distance parcourue entre 2 tics pour la roue B en mm*/ /*DIAMETRE_ROUE_B*PI/TIC_PAR_TOUR_B*/

#define NIV_PRECI_MM 1 /*Précision des déplacements du robot*/
#define NIV_PRECI_DEG 1 /*Précision des rotations du robot*/

// Liaison PC
Serial pc(SERIAL_TX, SERIAL_RX, 115200);

// Initialisation encodeur
QEI encodeur_a (D2, D3, NC, 1200, QEI::X4_ENCODING);
QEI encodeur_b (D4, D5, NC, 1200, QEI::X4_ENCODING);

Ticker timer;
DigitalIn way(PC_8,PullDown);
DigitalIn languette(D12,PullDown);

// Prototypes des fonctions
double* get_vitesses();
double* get_distances();
double get_angle(double dist_A, double dist_B);
double get_dist_moy(double dist_A, double dist_B);
double PID_vitA(double erreur_vitA);
double PID_vitB(double erreur_vitB);
double PID_distA(double erreur_distA);
double PID_distB(double erreur_distB);
double PID_angleA(double erreur_angleA);
double PID_angleB(double erreur_angleB);
double PID_distA2(double erreur_distA2);
double PID_distB2(double erreur_distB2);
double PID_angleA2(double erreur_angleA2);
double PID_angleB2(double erreur_angleB2);

// Coefficients PID pour asservissement en vitesse
const double KpvA = 0.00240; //0.00240 
const double TivA = 0.0000020; //0.000020 
const double TdvA = 0.0;

const double KpvB = 0.00245; // 0.00245 
const double TivB = 0.0000024; // 0.00024
const double TdvB = 0.0;

// Coefficients PID pour asservissement en position
const double KpdA = 5.5;//5.5 -> 20 cm;
const double TidA = 0.007; //0.007
const double TddA = 0.0;

const double KpdB = 5.5;//5.5 -> 20 cm;
const double TidB = 0.007; //0.007
const double TddB = 0.0;

// Coefficients PID pour asservissement en angle
const double KpaA = 8; //8
const double KiaA = 0.08;//0.08
const double KdaA = 0.0;

const double KpaB = 8; //8
const double KiaB = 0.08;//0.08
const double KdaB = 0.0;


// Coefficients PID pour asservissement en position2
const double KpdA2 = 0;//5.5 -> 20 cm;
const double TidA2 = 0.00; //0.007
const double TddA2 = 0.0;

const double KpdB2 = 0;//5.5 -> 20 cm;
const double TidB2 = 0.00; //0.007
const double TddB2 = 0.0;

// Coefficients PID pour asservissement en angle2
const double KpaA2 = 7; //8
const double KiaA2 = 0.005;//0.08
const double KdaA2 = 0.0;

const double KpaB2 = 6.8; //8
const double KiaB2 = 0.005;//0.08
const double KdaB2 = 0.0;

// Erreurs moteur A
double errPrev_vitA = 0, errSum_vitA = 0, errDif_vitA = 0;
double errPrev_distA = 0, errSum_distA = 0, errDif_distA = 0;

// Erreurs moteur B
double errPrev_vitB = 0, errSum_vitB = 0, errDif_vitB = 0;
double errPrev_distB = 0, errSum_distB = 0, errDif_distB = 0;

// Erreurs angle
double errPrev_angleA = 0, errSum_angleA = 0, errDif_angleA = 0;
double errPrev_angleB = 0, errSum_angleB = 0, errDif_angleB = 0;

// Erreurs distance2
double errPrev_distA2 = 0, errSum_distA2 = 0, errDif_distA2 = 0;
double errPrev_distB2 = 0, errSum_distB2 = 0, errDif_distB2 = 0;

// Erreurs angle2
double errPrev_angleA2 = 0, errSum_angleA2 = 0, errDif_angleA2 = 0;
double errPrev_angleB2 = 0, errSum_angleB2 = 0, errDif_angleB2 = 0;

// Tableau des vitesses, positions [0] moteur A / [1] moteur B et coord [0]--> X / [1]--> Y / [2] angle
double T_vitesses[2], T_distances[2], T_coord[3];
const uint16_t L = 1000; //Longueur tableau d'affichage
const uint8_t nbr_consigne = 1;
const uint8_t nbr_result_afficher = 6;

int main(){
    printf("\nbjr \r\n\n");

    moteur_init();      /*Initialisation des moteurs*/

    // Consigne
    double consigne_vit_A, consigne_vit_B;
    double consigne_dist; // = 200;
    double consigne_angle; // = 0;
    double T_consigne_dist[nbr_consigne] = {0.0};
    double T_consigne_angle[nbr_consigne] = {10.0};

    //Initialisation des variables
    int32_t i = 0, j = 0, k = 0, l = 0, p = 0, r = 0;
    double dist_A, dist_B, dist, vit_A, vit_B;
    double pwm_PID_A, pwm_PID_B;
    double erreur_vit_A, erreur_vit_B;
    double erreur_dist;// erreur_dist_A, erreur_dist_B;
    double temps, temps0;
    double tmps_start; //dif_temps;
    double erreur_angle, angle;
    double pid_distA, pid_distB, pid_angleA, pid_angleB;

    // Initionalise le tableau des vitesses avec que des zéros
    double T_aff[L][nbr_result_afficher];
    for(k = 0 ; k < L ; k++){
        for(l = 0 ; l < nbr_result_afficher ; l++){
            T_aff[k][l] = 0;
        }
    }

    //Mesure du temps au lancement du programme en µs
    temps0 = us_ticker_read();    
    for (p = 0; p < nbr_consigne; p++){
        consigne_dist = T_consigne_dist[p];
        consigne_angle = T_consigne_angle[p];
        while(1) {
            // Retourne la vitesse du robot
            get_vitesses();
            vit_A = T_vitesses[0];
            vit_B = T_vitesses[1];
            
            // Retourne la distance parcourue par le robot
            get_distances();
            dist_A = T_distances[0];
            dist_B = T_distances[1];    
            
            // Calcul la position du robot
            dist = get_dist_moy(dist_A, dist_B); 
            
            // Erreur angle en degré
            angle = get_angle(dist_A, dist_B);
            erreur_angle = consigne_angle - angle;            
            
            //Temps depuis le lancement du robot
            temps = us_ticker_read();   //Mesure de temps initiale en µs
            tmps_start = (temps - temps0)/1000000;
    
            // Erreur de distance
            erreur_dist = consigne_dist - dist;
    
            // Calcul de la consigne de vitesse à partir du PID en distance
            if(consigne_angle > 3){
                pid_distA = PID_distA2(erreur_dist);
                pid_distB = PID_distB2(erreur_dist);
                pid_angleA = PID_angleA2(erreur_angle);
                pid_angleB = PID_angleB2(erreur_angle);
            }
            else{
                pid_distA = PID_distA(erreur_dist);
                pid_distB = PID_distB(erreur_dist);
                pid_angleA = PID_angleA(erreur_angle);
                pid_angleB = PID_angleB(erreur_angle);
            }
            consigne_vit_A = pid_distA - pid_angleA;
            consigne_vit_B = pid_distB + pid_angleB;
    
            // Erreur de vitesse
            erreur_vit_A = consigne_vit_A - vit_A;
            erreur_vit_B = consigne_vit_B - vit_B;
    
            // Calcul de la consigne pour le moteur à partir du PID en position
            pwm_PID_A = PID_vitA(erreur_vit_A);
            pwm_PID_B = PID_vitB(erreur_vit_B);
            
            // Limite la vitesse des moteurs à 50%
            if((pwm_PID_A) >= 0.556f){
                pwm_PID_A = 0.556;
            }
            if((pwm_PID_B) >= 0.55f){
                pwm_PID_B = 0.55;
            }
            
            if((pwm_PID_A) <= -0.556f){
                pwm_PID_A = -0.556;
            }
            if((pwm_PID_B) <= -0.55f){
                pwm_PID_B = -0.55;
            }
            
            // Envoie la nouvelle commande de vitesse
            moteur_b(pwm_PID_B);
            moteur_a(pwm_PID_A);
    
            //Affichage des distances et des vitesses /!\ Couteux en temps, modifie le temps de réponse de l'asservissement /!\ Uniquement debug
                //printf("   dist_A = %.2f , dist_B = %.2f , vit_A = %.1f mm/s, vit_B = %.1f mm/s, temps = %.2f, i = %2d   \n\r   erreur_dist_A = %.1f mm/s, erreur_dist_B = %.1f mm/s erreur_vit_A = %.1f mm/s, erreur_vit_B = %.1f mm/s     \n\n\r", dist_A, dist_B, vit_A, vit_B, tmps_start, i, erreur_dist_A, erreur_dist_B, erreur_vit_A, erreur_vit_B);
                //printf(" A = %.2f,  B = %.2f, A = %.2f,  B = %.2f\n\r", pid_distA, pid_distB, pid_angleA, pid_angleB); 
                //printf("A = %.2f,  B = %.2f\n\r", consigne_vit_A, consigne_vit_B);
            if(i < L){
                T_aff[i][0] = dist;
                T_aff[i][1] = angle;
                T_aff[i][2] = pid_distA;
                T_aff[i][3] = pid_angleA;
                T_aff[i][4] = consigne_vit_A;
                T_aff[i][5] = consigne_vit_B;
            }
            i++;
            if(i == L){
                printf("Tableau rempli\r\n");
            }
            
            // Le robot s'arrête à + ou - de NIV_PRECI_MM de la commande de distance
            if ((erreur_dist < NIV_PRECI_MM) && (erreur_dist > -NIV_PRECI_MM)) {
                if ((erreur_angle < NIV_PRECI_DEG && erreur_angle > -NIV_PRECI_DEG)) {
                    moteur_a(0);
                    moteur_b(0);
                    break;
                }
            }
    
            // Arrete le robot après 8 secondes
            if (tmps_start > 4){
                moteur_a(0);
                moteur_b(0);
                break;
            }
            
            // Arrête les moteurs après 700mm
            if (dist > 700 || dist < -700) {
                moteur_a(0);
                moteur_b(0);
                break;
            }
            // Arrête les moteurs après rotation de 200°
            if (angle > 200 || angle < -200) {
                moteur_a(0);
                moteur_b(0);
                break;
            }
        }
        
        // Affiche les résultats après que le robot soit arreté
        printf("Consigne %d \n\r", p);
        for(j = 0; j < nbr_result_afficher; j++){
            printf("\n\rValeur %d \n\r", j+1);
            for(r = 0; r < L; r+=L/400){
                printf("%.2f, ", T_aff[r][j]);
            }
        }
        // Rempli le tableau d'affichage de 0
        for(k = 0 ; k < L ; k++){
            for(l = 0 ; l < nbr_result_afficher ; l++){
                T_aff[k][l] = 0;
            }
        }
        
        // Recommence au début du tableau d'affichage
        i = 0;
        
        temps0 = us_ticker_read();    
    }
}

double* get_vitesses(){
    int32_t tic_A, tic_B, dif_tic_A, dif_tic_B;
    double temps, dif_temps, dif_dist_A, dif_dist_B;
    double ddist, dangle, dx, dy, angle;
    
    //Mesure du nombres de tic de chaque roue
    tic_A = encodeur_a.getPulses();
    tic_B = encodeur_b.getPulses();
    angle = get_angle(T_distances[0], T_distances[1])*(PI/180);

    temps = us_ticker_read();   //Mesure de temps initiale en µs
    wait(0.005);     //"Pause" pour ralentir le programme --> Effectue les mesures tous les 5ms(200Hz) /!\ Modifier cette valeur modifie le coef intégrateur des PID /!\

    //Mesure de la différence de tic et de temps entre avant et après la "pause"
    dif_tic_A =  encodeur_a.getPulses() - tic_A;
    dif_tic_B =  encodeur_b.getPulses() - tic_B;
    dif_temps = (us_ticker_read() - temps)/1000000;

    //Distance parcourue pendant la "pause"
    dif_dist_A = dif_tic_A*DIST_PAR_TIC_A;
    dif_dist_B = dif_tic_B*DIST_PAR_TIC_B;

    // Calcul des petits déplacements en coordonnées pendant la "pause"
    ddist = (dif_dist_A + dif_dist_B)/2;
    dangle = ((dif_dist_A - dif_dist_B)/DISTANCE_ENTRE_ROUES)*(180/PI);
    dx = ddist*cos(angle);
    dy = ddist*sin(angle);
    
    //Tableau des coordonnées (x, y et angle)
    T_coord[0] = T_coord[0] + dx;
    T_coord[1] = T_coord[1] + dy;
    T_coord[2] = T_coord[2] + dangle;

    //Vitesse moyenne pendant la "pause"
    T_vitesses[0] = dif_dist_A/dif_temps;
    T_vitesses[1] = dif_dist_B/dif_temps;
    
    return T_vitesses;
}

double* get_distances(){
    int32_t tic_A, tic_B;
    
    //Mesure du nombres de tic de chaque roue²i
    tic_A = encodeur_a.getPulses();
    tic_B = encodeur_b.getPulses();
    
    //Distance parcourue depuis l'allumage du robot pour chaque roue
    T_distances[0] = tic_A*DIST_PAR_TIC_A;
    T_distances[1] = tic_B*DIST_PAR_TIC_B;
    
    return T_distances;
}

double get_angle(double dist_A, double dist_B){
    //return atan((dist_A-dist_B)/DISTANCE_ENTRE_ROUES)*(92);
    //return atan2((double)dist_A-dist_B,DISTANCE_ENTRE_ROUES)*(114.592);
    return fmod(((dist_A-dist_B)/DISTANCE_ENTRE_ROUES)*(180/PI), 360);
}

double get_dist_moy(double dist_A, double dist_B){    
    return (dist_A + dist_B)/2;
}

double PID_vitA(double erreur_vit){
    double P, I, D;
    errSum_vitA += erreur_vit;
    errDif_vitA = erreur_vit - errPrev_vitA;
    errPrev_vitA = erreur_vit;

    P = erreur_vit * KpvA;
    I = errSum_vitA * TivA;
    D = errDif_vitA * TdvA;
    return P + I + D;
}

double PID_vitB(double erreur_vit){
    double P, I, D;
    errSum_vitB += erreur_vit;
    errDif_vitB = erreur_vit - errPrev_vitB;
    errPrev_vitB = erreur_vit;

    P = erreur_vit * KpvB;
    I = errSum_vitB * TivB;
    D = errDif_vitB * TdvB;
    return P + I + D;
}

//PID pour le déplacement en ligne droite seul
double PID_distA(double erreur_dist){
    double P, I, D;
    errSum_distA += erreur_dist;
    errDif_distA = erreur_dist - errPrev_distA;
    errPrev_distA = erreur_dist;

    P = erreur_dist * KpdA;
    I = errSum_distA * TidA;
    D = errDif_distA * TddA;
    return P + I + D;
}

double PID_distB(double erreur_dist){
    double P, I, D;
    errSum_distB += erreur_dist;
    errDif_distB = erreur_dist - errPrev_distB;
    errPrev_distB = erreur_dist;

    P = erreur_dist * KpdB;
    I = errSum_distB * TidB;
    D = errDif_distB * TddB;
    return P + I + D;
}

double PID_angleA(double erreur_angle){
    double P, I, D;
    errSum_angleA += erreur_angle;
    errDif_angleA = erreur_angle - errPrev_angleA;
    errPrev_angleA = erreur_angle;

    P = erreur_angle * KpaA;
    I = errSum_angleA * KiaA;
    D = errDif_angleA * KdaA;
    return P + I + D;
}

double PID_angleB(double erreur_angle){

    double P, I, D;
    errSum_angleB += erreur_angle;
    errDif_angleB = erreur_angle - errPrev_angleB;
    errPrev_angleB = erreur_angle;

    P = erreur_angle * KpaB;
    I = errSum_angleB * KiaB;
    D = errDif_angleB * KdaB;
    return P + I + D;
}

// PID pour la rotation seule
double PID_distA2(double erreur_dist2){
    double P, I, D;
    errSum_distA2 += erreur_dist2;
    errDif_distA2 = erreur_dist2 - errPrev_distA2;
    errPrev_distA2 = erreur_dist2;

    P = erreur_dist2 * KpdA2;
    I = errSum_distA2 * TidA2;
    D = errDif_distA2 * TddA2;
    return P + I + D;
}

double PID_distB2(double erreur_dist2){
    double P, I, D;
    errSum_distB2 += erreur_dist2;
    errDif_distB2 = erreur_dist2 - errPrev_distB2;
    errPrev_distB2 = erreur_dist2;

    P = erreur_dist2 * KpdB2;
    I = errSum_distB2 * TidB2;
    D = errDif_distB2 * TddB2;
    return P + I + D;
}

double PID_angleA2(double erreur_angle2){
    double P, I, D;
    errSum_angleA2 += erreur_angle2;
    errDif_angleA2 = erreur_angle2 - errPrev_angleA2;
    errPrev_angleA2 = erreur_angle2;

    P = erreur_angle2 * KpaA2;
    I = errSum_angleA2 * KiaA2;
    D = errDif_angleA2 * KdaA2;
    return P + I + D;
}

double PID_angleB2(double erreur_angle2){
    double P, I, D;
    errSum_angleB2 += erreur_angle2;
    errDif_angleB2 = erreur_angle2 - errPrev_angleB2;
    errPrev_angleB2 = erreur_angle2;

    P = erreur_angle2 * KpaB2;
    I = errSum_angleB2 * KiaB2;
    D = errDif_angleB2 * KdaB2;
    return P + I + D;
}