/// @file Watchdog.cpp provides the interface to the Watchdog module
///
/// This provides basic Watchdog service for the mbed. You can configure
/// various timeout intervals that meet your system needs. Additionally,
/// it is possible to identify if the Watchdog was the cause of any 
/// system restart.
/// 
/// Adapted from Simon's Watchdog code from http://mbed.org/forum/mbed/topic/508/


#include "mbed.h"
#include "Watchdog.h"

#ifdef FEATURE_COMMON_PAL
#include "mbed_trace.h"
#define TRACE_GROUP "WDT"
#else
#define tr_debug(format, ...) debug_if(_debug_trace_on, format "\n", ## __VA_ARGS__)
#define tr_info(format, ...)  debug_if(_debug_trace_on, format "\n", ## __VA_ARGS__)
#define tr_warn(format, ...)  debug_if(_debug_trace_on, format "\n", ## __VA_ARGS__)
#define tr_error(format, ...) debug_if(_debug_trace_on, format "\n", ## __VA_ARGS__)
#endif

/// Watchdog gets instantiated at the module level
Watchdog::Watchdog() {
    // capture the cause of the previous reset
    wdreset = __HAL_RCC_GET_FLAG(RCC_FLAG_IWDGRST);
    //Clear source Reset Flag
    __HAL_RCC_CLEAR_RESET_FLAGS();
}

/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float timeout) {
       
    #define LsiFreq (45000)
    uint16_t PrescalerCode;
    uint16_t Prescaler;
    uint16_t ReloadValue;
    float Calculated_timeout;
    
    if ((timeout * (LsiFreq/4)) < 0x7FF) {
        PrescalerCode = IWDG_PRESCALER_4;
        Prescaler = 4;
    }
    else if ((timeout * (LsiFreq/8)) < 0xFF0) {
        PrescalerCode = IWDG_PRESCALER_8;
        Prescaler = 8;
    }
    else if ((timeout * (LsiFreq/16)) < 0xFF0) {
        PrescalerCode = IWDG_PRESCALER_16;
        Prescaler = 16;
    }
    else if ((timeout * (LsiFreq/32)) < 0xFF0) {
        PrescalerCode = IWDG_PRESCALER_32;
        Prescaler = 32;
    }
    else if ((timeout * (LsiFreq/64)) < 0xFF0) {
        PrescalerCode = IWDG_PRESCALER_64;
        Prescaler = 64;
    }
    else if ((timeout * (LsiFreq/128)) < 0xFF0) {
        PrescalerCode = IWDG_PRESCALER_128;
        Prescaler = 128;
    }
    else {
        PrescalerCode = IWDG_PRESCALER_256;
        Prescaler = 256;
    }
    
    // specifies the IWDG Reload value. This parameter must be a number between 0 and 0x0FFF.
    ReloadValue = (uint32_t)(timeout * (LsiFreq/Prescaler));
    
    Calculated_timeout = ((float)(Prescaler * ReloadValue)) / LsiFreq;
    tr_debug("WATCHDOG set with prescaler:%d reload value: 0x%X - timeout:%f\n",Prescaler, ReloadValue, Calculated_timeout);
    
    IWDG->KR = 0x5555; //Disable write protection of IWDG registers      
    IWDG->PR = PrescalerCode;      //Set PR value      
    IWDG->RLR = ReloadValue;      //Set RLR value      
    IWDG->KR = 0xAAAA;    //Reload IWDG      
    IWDG->KR = 0xCCCC;    //Start IWDG - See more at: http://embedded-lab.com/blog/?p=9662#sthash.6VNxVSn0.dpuf       
 
    Pet();
}

/// "Pet", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Pet() { 
    IWDG->KR = 0xAAAA;         //Reload IWDG - See more at: http://embedded-lab.com/blog/?p=9662#sthash.6VNxVSn0.dpuf
}

/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset() {
    return wdreset;
}
