// This provides a reset supervisor service for the mbed. You can configure
// various timeout intervals for the watchdog that meet your system needs. Additionally,
// it is possible to identify if the Supervisor was the cause of any 
// system restart, permitting the application code to take appropriate
// behavior.
// 
// Adapted from Simon's watchdog code from http://mbed.org/forum/mbed/topic/508/
//
//
#ifndef RESETSUPERVISOR_H
#define RESETSUPERVISOR_H
#include "mbed.h"

// The Supervisor class provides the interface to the Supervisor feature
//
// Embedded programs, by their nature, are usually unattended. If things
// go wrong, it is usually important that the system attempts to recover.
// Aside from robust software, a hardware watchdog can monitor the
// system and initiate a system reset when appropriate.
//
// This Supervisor is patterned after one found elsewhere on the mbed site,
// however this one also provides a method for the application software
// to determine the cause of the reset - watchdog or otherwise.
//
// example:
// @code
// Supervisor wd;
//
// ...
// main() {
//    if (wd.SupervisorCausedReset())
//        pc.printf("Supervisor caused reset.\r\n");
//      
//    wd.Configure(3.0);       // sets the timeout interval
//    for (;;) {
//         wd.Update();       // kick the dog before the timeout
//         // do other work
//    }
// }
// @endcode
//
class Supervisor {
public:
    // Create a Supervisor object
    //
    // example:
    // @code
    // Supervisor wd;    // placed before main
    // @endcode
    Supervisor();
    
    // Configure the timeout for the Supervisor
    //
    // This configures the Supervisor service and starts it. It must
    // be serviced before the timeout, or the system will be restarted.
    //
    // example:
    // @code
    //     ...
    //     wd.initWD(1.4);  // configure for a 1.4 second timeout
    //     ...
    // @endcode
    //
    // @param[in] timeout in seconds, as a floating point number
    // @returns none
    //
    void initWD(float timeout);
    
    // Service the Supervisor so it does not cause a system reset
    //
    // example:
    // @code
    //    wd.pingWD();
    // @endcode
    // @returns none
    void pingWD();
     
    /* resetReason explains what exactly caused the system reset
     *
     * example:
     * 
     *    switch(wd.ResetReason()){
     *        case RCC_FLAG_BORRST:
     *        .
     *        .
     *        break,
     *    }
     *
     *  returns:
     *   RCC_FLAG_BORRST    BOR reset
     *   RCC_FLAG_OBLRST    OBLRST reset
     *   RCC_FLAG_PINRST    Pin reset
     *   RCC_FLAG_FWRST     FIREWALL reset
     *   RCC_FLAG_SFTRST    Software reset
     *   RCC_FLAG_IWDGRST   Independent Supervisor reset
     *   RCC_FLAG_WWDGRST   Window Supervisor reset
     *   RCC_FLAG_LPWRRST   Low Power reset
    */
    uint8_t resetReason();
    
    /* doReset causes the device to perform a software reset
     *
     *  example:
     *
     *      wd.doReset();
     *
     *  returns:
     *      N/A
     */
     void doReset();  
    
private:
    uint8_t rstRsn;

};

#endif // WATCHDOG_H