#include "pixart_lcm.h"



typedef struct
{
    SPI                *pSPI ;
    DigitalOut          *pCSB;
    DigitalOut          *pRSTB;
    DigitalOut          *pRS;
} pixart_lcm_state_s;

static pixart_lcm_state_s g_state ;


#define LOW 0
#define HIGH 1
#define PIN_LCM_RS      g_state.pRS
#define PIN_LCM_RSTB    g_state.pRSTB
#define PIN_LCM_CSB     g_state.pCSB
#define digitalWrite(pin,level) *pin =  level
#define LCM_RS_LO       digitalWrite(PIN_LCM_RS,LOW)        
#define LCM_RS_HI       digitalWrite(PIN_LCM_RS,HIGH)   
#define LCM_RSTB_LO     digitalWrite(PIN_LCM_RSTB,LOW)
#define LCM_RSTB_HI     digitalWrite(PIN_LCM_RSTB,HIGH)
#define LCM_CSB_LO      digitalWrite(PIN_LCM_CSB,LOW)
#define LCM_CSB_HI      digitalWrite(PIN_LCM_CSB,HIGH)

#define I2C_ADDRESS    (0x73 << 1)
#define delayMicroseconds(us) wait_us(us)
#define delay(ms) wait_ms(ms)

//-----------------------------------------------------------------------
unsigned char hex2dec_nibble(unsigned char hex_nibble)
{
    unsigned char dec;

    switch(hex_nibble)
    {
    case 0xA: dec=10;   break;
    case 0xB: dec=11;   break;
    case 0xC: dec=12;   break;
    case 0xD: dec=13;   break;
    case 0xE: dec=14;   break;
    case 0xF: dec=15;   break;
    default:    dec=hex_nibble; break;
    }
    
    return (dec);
}

unsigned int hex2dec_word(unsigned int hex_word)
{
    unsigned char dec_nb[4];
    unsigned char nibble3=(hex_word>>12)&0x000f;
    unsigned char nibble2=(hex_word>>8)&0x000f;
    unsigned char nibble1=(hex_word>>4)&0x000f;
    unsigned char nibble0=hex_word&0x000f;

    dec_nb[3]=hex2dec_nibble(nibble3);
    dec_nb[2]=hex2dec_nibble(nibble2);
    dec_nb[1]=hex2dec_nibble(nibble1);
    dec_nb[0]=hex2dec_nibble(nibble0);
    return ((dec_nb[3]<<12)+(dec_nb[2]<<8)+(dec_nb[1]<<4)+dec_nb[0]);
}

void LCM_WriteCom(unsigned char Command)
{
    LCM_CSB_LO;
    LCM_RS_LO;
    //SPI.transfer(Command);
    g_state.pSPI->write(Command);
    LCM_CSB_HI;

    delayMicroseconds(30);
}

void LCM_WriteData(unsigned char Ascii)
{
    LCM_CSB_LO;
    LCM_RS_HI;
    //SPI.transfer(Ascii);
    g_state.pSPI->write(Ascii);
    LCM_CSB_HI;
    
    delayMicroseconds(30); 
}

void LCM_Init(void)
{
    LCM_RSTB_LO;
    delay(3);
    LCM_RSTB_HI;
    delay(20);
    LCM_WriteCom(0x30); //wake up
    delay(3);
    LCM_WriteCom(0x30); //wake up
    LCM_WriteCom(0x30); //wake up   
    LCM_WriteCom(0x39); //function set
    LCM_WriteCom(0x14); //internal osc frequency
    LCM_WriteCom(0x56); //Contrast set
    LCM_WriteCom(0x6D); //follower control
    LCM_WriteCom(0x75); //contrast//
    LCM_WriteCom(0x0C); //display on
    LCM_WriteCom(0x06); //entry mode
    LCM_WriteCom(0x01); //clear
    
    delay(10);
}

void LCM_Clear(void)
{
    LCM_WriteCom(0x01);

    delay(2);
}

void LCM_SetPosition(unsigned char line, unsigned char position)//line=1 or 2; position=1~16
{
    unsigned char address;
    
    address = ((line-1) * 0x40) + (position-1);
    address = 0x80 + (address & 0x7F);
    
    LCM_WriteCom(address);
}

void LCM_DisplayString(unsigned char line, unsigned char position, const char *ptr)
{
    LCM_SetPosition(line,position);
    
    while (*ptr)  
    {
            LCM_WriteData(*ptr++);
    }
}

void LCM_DisplayDecimal(unsigned char line, unsigned char position, unsigned int hex_word, unsigned char digits)
{
    unsigned char sign;//0:positive, 1:negative
    unsigned int dec_num;
    unsigned char digit[5];
    signed char ii;
    
    if(hex_word & 0x8000)   
        sign=1;
    else    
        sign=0;

    if(sign==1) 
        hex_word=~hex_word+1;

    dec_num=hex2dec_word(hex_word);
    digit[4]=dec_num/10000;
    digit[3]=(dec_num%10000)/1000;
    digit[2]=(dec_num%1000)/100;
    digit[1]=(dec_num%100)/10;
    digit[0]=dec_num%10;

    LCM_SetPosition(line,position);

    if(sign==1) 
        LCM_WriteData('-');
    else
        LCM_WriteData('+');

    for(ii=(digits-1);ii>=0;ii--)
    {
        LCM_WriteData(digit[ii] | 0x30);//decimal to ascii
    }

}

void LCM_ClearLine(unsigned char line)//    line: 1 or 2
{
    LCM_DisplayString(line,1,"                ");
}

void LCM_DisplayString_Reset(void)
{
  LCM_DisplayString(1,1,"SHAFT"); LCM_DisplayString(1,8,"U/D");LCM_DisplayDecimal(1,12,0x000,4);
  LCM_DisplayString(2,1,"SPRING");LCM_DisplayString(2,8,"P/R");LCM_DisplayDecimal(2,12,0x000,4);    
}

void LCM_DisplayString_Boot(boolean sen_status)
{
  LCM_DisplayString(1,1,"PixArt Shaft EVK");    
  LCM_DisplayString(2,1,"PAT9125 FW V2.40");
  delay(2000);
  
  LCM_ClearLine(1);
  LCM_ClearLine(2);
  
  if(sen_status == true)
  {
    LCM_DisplayString_Reset();
  }
  else
  {
    LCM_DisplayString(2,1,"Read Sensor Fail");
    //while(1);//stop here if read sensor fail as a warning.
  }
}

//-----------------------------------------------------------------------

pixart_lcm::pixart_lcm(SPI *pSPI, DigitalOut *pCSB, DigitalOut *pRSTB, DigitalOut *pRS)
{
    g_state.pSPI = pSPI;
    g_state.pCSB = pCSB;
    g_state.pRSTB = pRSTB;
    g_state.pRS = pRS;
    *g_state.pRS =  1;
    LCM_Init();
}

void pixart_lcm::LCM_DisplayString(unsigned char line, unsigned char position, const char *ptr)
{
    ::LCM_DisplayString(line, position, ptr);
}
void pixart_lcm::LCM_DisplayDecimal(unsigned char line, unsigned char position, unsigned int hex_word, unsigned char digits)
{
    ::LCM_DisplayDecimal(line, position, hex_word, digits);
}
void pixart_lcm::LCM_DisplayString_Reset(void)
{
    ::LCM_DisplayString_Reset();
}
void pixart_lcm::LCM_DisplayString_Boot(boolean sen_status)
{    
    ::LCM_DisplayString_Boot(sen_status);
}