#define SPImode                                     //Used to check if the sensor is in I2C mode or SPI mode.

//=========================================================================
//Communication pinouts for serial COM port, SPI, and interrupts
//=========================================================================
static Serial pc(USBTX, USBRX);                     //PC comm
static SPI spi(p23, p24, p25);                      //mosi, miso, sclk
static DigitalOut cs(p22);                          //chip select
static DigitalOut shutdown(p20);                    //Shutdown pin
static DigitalOut IO_sel(p19);                      //IO interface selection pin (I2C and SPI)


//=========================================================================
//Variables and arrays used for communications and data storage
//=========================================================================
int8_t deltaX, deltaY;                              //Stores the value of one individual motion report.
int totalX, totalY = 0;                          //Stores the total deltaX and deltaY moved during runtime.


//=========================================================================
//Functions used to communicate with the sensor and grab/print data
//=========================================================================
uint8_t readRegister(uint8_t addr);
//This function takes an 8-bit address in the form 0x00 and returns an 8-bit value in the form 0x00.

void writeRegister(uint8_t addr, uint8_t data);
//This function takes an 8-bit address and 8-bit data. Writes the given data to the given address.

void load(const uint8_t array[][2], uint8_t arraySize);
//Takes an array of registers/data (found in registerArrays.h) and their size and writes in all the values.

void grabData(void);
//Grabs the deltaX and deltaY information from the proper registers and formats it into the proper format.

void printData(void);
//Prints the data out to a serial terminal.





//=========================================================================
//Functions definitions
//=========================================================================
uint8_t readRegister(uint8_t addr)
{
    cs = 0;                                 //Set chip select low/active
    addr = addr & 0x7F;                     //Set MSB to 0 to indicate read operation
    spi.write(addr);                        //Write the given address
    wait_us(5);
    uint8_t data_read = spi.write(0x00);    //Throw dummy byte after sending address to receieve data
    wait_us(30);
    cs = 1;                                 //Set chip select back to high/inactive
    return data_read;                       //Returns 8-bit data from register
}


//=========================================================================
void writeRegister(uint8_t addr, uint8_t data)
{
    cs = 0;                         //Set chip select low/active
    addr = addr | 0x80;             //Set MSB to 1 to indicate write operation
    spi.write(addr);                //Write the given address
    spi.write(data);                //Write the given data
    cs = 1;                         //Set chip select back to high/inactive
    wait_us(30);                    //Wait time between write commands.
    
    //pc.printf("R:%2X, D:%2X\n\r", addr, readRegister(addr));
            //Uncomment this line for debugging. Prints every register write operation.
}


//=========================================================================
void load(const uint8_t array[][2], uint8_t arraySize)
{
    for(uint8_t q = 0; q < arraySize; q++)
    {
        writeRegister(array[q][0], array[q][1]);    //Writes the given array of registers/data.
    }
}


//=========================================================================
void grabData(void)
{
    deltaX = readRegister(0x03);        //Grabs data from the proper registers.
    deltaY = readRegister(0x04);
    writeRegister(0x02, 0x00);          //Clear EVENT and motion registers.
}


//=========================================================================
void printData(void)
{
    if((deltaX != 0) || (deltaY != 0))      //If there is deltaX or deltaY movement, print the data.
    {
        totalX += deltaX;
        totalY += deltaY;
        
        pc.printf("deltaX: %d\t\t\tdeltaY: %d\n\r", deltaX, deltaY);    //Prints each individual count of deltaX and deltaY.
        pc.printf("X-axis Counts: %d\t\tY-axis Counts: %d\n\r", totalX, totalY);  //Prints the total movement made during runtime.
    }
    
    deltaX = 0;                             //Resets deltaX and Y values to zero, otherwise previous data is stored until overwritten.
    deltaY = 0;
}
