#define SPImode

//=========================================================================
//Communication pinouts for serial COM port, SPI, and interrupts
//=========================================================================
static Serial pc(USBTX, USBRX);                     //PC comm
static SPI spi(p23, p24, p25);                      //mosi, miso, sclk
static DigitalOut cs(p22);                          //chip select


//=========================================================================
//Variables and arrays used for communications and data storage
//=========================================================================
int8_t deltaX_low, deltaY_low;                      //Stores the low-bits of movement data.
int16_t deltaX_high, deltaY_high;                   //Stores the high-bits of movement data.
int16_t deltaX, deltaY;                             //Stores the combined value of low and high bits.
int16_t totalX, totalY = 0;                         //Stores the total deltaX and deltaY moved during runtime.


//=========================================================================
//Functions used to communicate with the sensor and grab/print data
//=========================================================================
uint8_t readRegister(uint8_t addr);
//This function takes an 8-bit address in the form 0x00 and returns an 8-bit value in the form 0x00.

void writeRegister(uint8_t addr, uint8_t data);
//This function takes an 8-bit address and 8-bit data. Writes the given data to the given address.

void load(const uint8_t array[][2], uint8_t arraySize);
//Takes an array of registers/data (found in registerArrays.h) and their size and writes in all the values.

void grabData(void);
//Grabs the deltaX and deltaY information from the proper registers and formats it into the proper format.

void printData(void);
//Prints the data out to a serial terminal.





//=========================================================================
//Functions definitions
//=========================================================================
uint8_t readRegister(uint8_t addr)
{
    cs = 0;                                 //Set chip select low/active
    addr = addr & 0x7F;                     //Set MSB to 0 to indicate read operation
    spi.write(addr);                        //Write the given address
    wait_us(1);                             //Add a tiny delay after sending address for some internal cycle timing.
    uint8_t data_read = spi.write(0x00);    //Throw dummy byte after sending address to receieve data
    cs = 1;                                 //Set chip select back to high/inactive
    return data_read;                       //Returns 8-bit data from register
}


//=========================================================================
void writeRegister(uint8_t addr, uint8_t data)
{
    cs = 0;                         //Set chip select low/active
    addr = addr | 0x80;             //Set MSB to 1 to indicate write operation
    spi.write(addr);                //Write the given address
    spi.write(data);                //Write the given data
    cs = 1;                         //Set chip select back to high/inactive
    
    //pc.printf("R:%2X, D:%2X\n\r", addr, readRegister(addr));
            //Uncomment this line for debugging. Prints every register write operation.
}


//=========================================================================
void load(const uint8_t array[][2], uint8_t arraySize)
{
    for(uint8_t q = 0; q < arraySize; q++)
    {
        writeRegister(array[q][0], array[q][1]);    //Writes the given array of registers/data.
    }
}


//=========================================================================
void grabData(void)
{
    deltaX_low = readRegister(0x03);        //Grabs data from the proper registers.
    deltaY_low = readRegister(0x04);
    deltaX_high = (readRegister(0x12)<<4) & 0xF00;      //Grabs data and shifts it to make space to be combined with lower bits.
    deltaY_high = (readRegister(0x12)<<8) & 0xF00;
    
    if(deltaX_high & 0x800)
        deltaX_high |= 0xf000; // 12-bit data convert to 16-bit (two's comp)
        
    if(deltaY_high & 0x800)
        deltaY_high |= 0xf000; // 12-bit data convert to 16-bit (2's comp)
        
    deltaX = deltaX_high | deltaX_low;      //Combined the low and high bits.
    deltaY = deltaY_high | deltaY_low;
}


//=========================================================================
void printData(void)
{
    if((deltaX != 0) || (deltaY != 0))      //If there is deltaX or deltaY movement, print the data.
    {
        totalX += deltaX;
        totalY += deltaY;
        
        pc.printf("deltaX: %d\t\t\tdeltaY: %d\n\r", deltaX, deltaY);    //Prints each individual count of deltaX and deltaY.
        pc.printf("X-axis Counts: %d\t\t\tY-axis Counts: %d\n\r", totalX, totalY);  //Prints the total movement made during runtime.
    }
    
    deltaX = 0;                             //Resets deltaX and Y values to zero, otherwise previous data is stored until overwritten.
    deltaY = 0;
}
