#define I2C_Slave_ID 0x33                           //Slave ID of the PMT9123 sensor.

//=========================================================================
//Communication pinouts for serial COM port, I2C, and interrupts
//=========================================================================
static Serial pc(USBTX, USBRX);                     //PC comm
static I2C i2c(p26, p27);                           //SDA, SCL
static DigitalOut NRESET(p25);                      //Laser diode pin.

//=========================================================================
//Variables and arrays used for communications and data storage
//=========================================================================
int8_t deltaX_low, deltaY_low;                      //Stores the low-bits of movement data.
int16_t deltaX_high, deltaY_high;                   //Stores the high-bits of movement data.
int16_t deltaX, deltaY;                             //Stores the combined value of low and high bits.
int16_t totalX, totalY = 0;                         //Stores the total deltaX and deltaY moved during runtime.


//=========================================================================
//Functions used to communicate with the sensor and grab/print data
//=========================================================================
uint8_t readRegister(uint8_t addr);
//This function takes an 8-bit address in the form 0x00 and returns an 8-bit value in the form 0x00.

void writeRegister(uint8_t addr, uint8_t data);
//This function takes an 8-bit address and 8-bit data. Writes the given data to the given address.

void startup(void);
//Starts up the sensor to prepare it after receiving power.

void initialize(void);
//Runs the initialization sequence for the sensor.

void load(const uint8_t array[][2], uint8_t arraySize);
//Takes an array of registers/data (found in registerArrays.h) and their size and writes in all the values.

void grabData(void);
//Grabs the deltaX and deltaY information from the proper registers and formats it into the proper format.

void printData(void);
//Prints the data out to a serial terminal.





//=========================================================================
//Functions definitions
//=========================================================================
uint8_t readRegister(uint8_t addr)
{
    uint8_t data;
    
    i2c.write((I2C_Slave_ID << 1), (const char*)&addr, 1, 0);   //Send the address to the chip
    wait_us(1);
    i2c.read((I2C_Slave_ID << 1), (char*)&data, 1, 0);          //Send the memory address where you want to store the read data
    return(data);
}


//=========================================================================
void writeRegister(uint8_t addr, uint8_t data)
{
    char data_write[2];             //Create an array to store the address/data to pass them at the same time.
    
    data_write[0] = addr;           //Store the address in the first byte
    data_write[1] = data;           //Store the data in the second byte
    i2c.write((I2C_Slave_ID << 1), data_write, 2, 0);   //Send both over at once
    
    pc.printf("R:%2X, D:%2X\n\r", addr, readRegister(addr));
            //Uncomment this line for debugging. Prints every register write operation.
}


//=========================================================================
void startup(void)
{
    NRESET = 0;                         //Drive NRESET low for 20us to reset the device.
    wait_us(20);
    NRESET = 1;
    wait_ms(1.5);
}


//=========================================================================
void initialize(void)
{
    writeRegister(0x41, 0xBA);          //Disables register write protection.
    wait_us(300);                       //Delay 300us for timing purposes. (necessary)
    
    writeRegister(0x1D, 0x00);          //Clears the "what mode are we in" register.
    wait_ms(10);                        //Delay 10ms for timing purposes. (necessary)
    
    readRegister(0x02);                 //Clears motion bit and motion storage registers.
    readRegister(0x03);
    readRegister(0x04);
    readRegister(0x05);
}


//=========================================================================
void load(const uint8_t array[][2], uint8_t arraySize)
{
    for(uint8_t q = 0; q < arraySize; q++)
    {
        writeRegister(array[q][0], array[q][1]);    //Writes the given array of registers/data.
    }
}


//=========================================================================
void grabData(void)
{
    deltaX_low = readRegister(0x03);        //Grabs data from the proper registers.
    deltaY_low = readRegister(0x04);
    deltaX_high = (readRegister(0x05)<<4) & 0xF00;      //Grabs data and shifts it to make space to be combined with lower bits.
    deltaY_high = (readRegister(0x05)<<8) & 0xF00;
    
    if(deltaX_high & 0x800)
        deltaX_high |= 0xf000; // 12-bit data convert to 16-bit (two's comp)
        
    if(deltaY_high & 0x800)
        deltaY_high |= 0xf000; // 12-bit data convert to 16-bit (2's comp)
        
    deltaX = deltaX_high | deltaX_low;      //Combined the low and high bits.
    deltaY = deltaY_high | deltaY_low;
}


//=========================================================================
void printData(void)
{
    if((deltaX != 0) || (deltaY != 0))      //If there is deltaX or deltaY movement, print the data.
    {
        totalX += deltaX;
        totalY += deltaY;
        
        pc.printf("deltaX: %d\t\t\tdeltaY: %d\n\r", deltaX, deltaY);    //Prints each individual count of deltaX and deltaY.
        pc.printf("X-axis Counts: %d\t\tY-axis Counts: %d\n\r", totalX, totalY);  //Prints the total movement made during runtime.
    }
    
    deltaX = 0;                             //Resets deltaX and Y values to zero, otherwise previous data is stored until overwritten.
    deltaY = 0;
}
