//=========================================================================
//Communication pinouts for serial COM port, SPI, and interrupts
//=========================================================================
static Serial pc(USBTX, USBRX);                     //PC serial communication.
static SPI spi(p23, p24, p25);                      //MOSI, MISO, SCLK  (because this is 3-Wire SPI, we tie MOSI and MISO together on the SDIO pin, but separate them when we get to the MCU).
static DigitalOut cs(p22);                          //Chip select pin.
static DigitalOut LDP(p20);                         //Laser diode pin.


//=========================================================================
//Variables and arrays used for communications and data storage
//=========================================================================
int16_t deltaX = 0;                                 //Stores X-axis output data.
int16_t deltaY = 0;                                 //Stores Y-axis output data.
int16_t totalX, totalY = 0;                         //Stores the total deltaX and deltaY moved during runtime.
uint8_t frameCounter = 0;                           //Looping variable to track when we have grabbed 8 frames of data.
uint16_t imageQuality[8];                           //Stores 8 frames of image data to determine when quality changes to a point where we need to switch modes (LED/laser).
uint16_t imageQuality_total = 0;                    //Stores the sum of all raw values from the 8 frames from imageQuality[8].
uint8_t mode = 0;                                   //Modes: 0 = Laser, 1 = LED.    Laser is default.
#define laser2LED_threshold 0x700                   //Stores threshold level for when we need to swap to LED mode.
#define LED2laser_threshold 0x500                   //Stores threshold level for when we need to swap to laser mode.

uint8_t currentBank;                                //Stores current register bank.


//=========================================================================
//Functions used to communicate with the sensor and grab/print data
//=========================================================================
uint8_t readRegister(uint8_t addr);
//This function takes an 8-bit address in the form 0x00 and returns an 8-bit value in the form 0x00.

void writeRegister(uint8_t addr, uint8_t data);
//This function takes an 8-bit address and 8-bit data. Writes the given data to the given address.

void load(const uint8_t array[][2], uint8_t arraySize);
//Takes an array of registers/data (found in registerArrays.h) and their size and writes in all the values.

void checkMode(void);
//This function checks the image quality of each frame and switches between LED and laser modes when necessary.

void grabData(void);
//Takes data from the sensor and stores it into variables deltaX and deltaY.

void printData(void);
//Prints the deltaX and deltaY values to a serial terminal whenever the values change.





//=========================================================================
//Functions definitions
//=========================================================================
uint8_t readRegister(uint8_t addr)
{
    cs = 0;                                 //Set chip select low/active
    addr = addr & 0x7F;                     //Set MSB to 0 to indicate read operation
    spi.write(addr);                        //Write the given address
    wait_ms(1);
    uint8_t data_read = spi.write(0x00);    //Throw dummy byte after sending address to receieve data
    cs = 1;                                 //Set chip select back to high/inactive
    return data_read;                       //Returns 8-bit data from register
}


//=========================================================================
void writeRegister(uint8_t addr, uint8_t data)
{
    cs = 0;                         //Set chip select low/active
    addr = addr | 0x80;             //Set MSB to 1 to indicate write operation
    spi.write(addr);                //Write the given address
    spi.write(data);                //Write the given data
    cs = 1;                         //Set chip select back to high/inactive
    
    if(addr == 0x7F)                //You can't read register 0x7F for the current bank so we store the value before writing it
    {
        currentBank = data;         //Store the current bank for printing later.
        //pc.printf("B:%2X, R:%2X, D:%2X\n\r", currentBank, (addr & 0x7F), currentBank);
            //Uncomment this line and the other print line below for debugging. Prints every register bank change.
    }
    
    else
    {
        //pc.printf("B:%2X, R:%2X, D:%2X\n\r", currentBank, (addr & 0x7F), readRegister(addr));
            //Uncomment this line and the other print line above for debugging. Prints every register write operation.
    }
}


//=========================================================================
void load(const uint8_t array[][2], uint8_t arraySize)
{
    for(uint8_t q = 0; q < arraySize; q++)
    {
        writeRegister(array[q][0], array[q][1]);    //Writes the given array of registers/data.
    }
}


//=========================================================================
void checkMode(void)
{
    uint16_t data_msb, data_lsb = 0;                 //These variables store the high and low bits of image quality data.
    imageQuality_total = 0;                         //This variable holds the sum of 8 frames of data.
    
    data_msb = readRegister(0x75);                  //Reads upper 8 bits of image quality.
    data_lsb = readRegister(0x76);                  //Reads lower 8 bits of image quality.
    imageQuality[frameCounter] = data_msb*256 + data_lsb;   //Combines the low/high bits to be a single element in this array.
    
    if(frameCounter == 7)                           //We check image quality every EIGHT samples before determining if a mode-switch is necessary.
    {
        for(int i=0; i<8; i++)                      //Sums up 8 frames of image quality data into 1 variable.
        {
            imageQuality_total = imageQuality_total + imageQuality[i];
        }
        
        if(mode == 1 && imageQuality_total < LED2laser_threshold)       //Check the condition for when we need to change to laser.
        {
            load(modeLaser, modeLaser_size);        //Loads the register array that has the settings for laser mode.
            mode = 0;                               //Sets the mode-tracker to be 0 to reflect laser mode.
            LDP = 0;                                //Sets the laser diode GPIO pin to be LOW (active because of PMOS pull-up).
            wait_ms(40);                            //Delay for timing.
            writeRegister(0x03, 0x00);              //Unlisted...
            
            //pc.printf("Image Quality: %4X\n\r", imageQuality_total);
            //pc.printf("Now in laser mode. Value of 'mode': %1X\n\r", mode);                
            //Uncomment these lines and the other print lines below for debugging. Prints image quality and tracks mode-switches.
        }
        
        if(mode == 0 && imageQuality_total < laser2LED_threshold)  //Check the condition for when we need to change to LED.
        {
            load(modeLED, modeLED_size);            //Loads the register array that has the settings for LED mode.
            mode = 1;                               //Sets the mode-tracker to be 1 to reflect LED mode.
            LDP = 1;                                //Sets the laser diode GPIO pin to be HIGH (inactive because of PMOS pull-up).
            wait_ms(40);                            //Delay for timing.
            writeRegister(0x03, 0x00);              //Unlisted...
            
            //pc.printf("Image Quality: %4X\n\r", imageQuality_total);
            //pc.printf("Now in LED mode. Value of 'mode': %1X\n\r", mode);
            //Uncomment these lines and the other print lines above for debugging. Prints image quality and tracks mode-switches.
        }
    }
    
    frameCounter = (frameCounter + 1) & 0x07;       //This variable loops from zero to 7 without using a looper or variable. Every time this function is called, it moves up 1 towards 7 or resets to zero.
}


//=========================================================================
void grabData(void)
{
    int16_t deltaX_high, deltaX_low = 0;            //These four variables store the low/high bits of the deltaX and deltaY data.
    int16_t deltaY_high, deltaY_low = 0;
    
    deltaX_low = (int16_t)readRegister(0x03);       //Grabbing the data from registers.
    deltaY_low = (int16_t)readRegister(0x04);
    deltaX_high = ((int16_t)readRegister(0x11))<<8; //Shifts high bits left by 8 so that we can combine high and low bytes together.
    deltaY_high = ((int16_t)readRegister(0x12))<<8;
    
    deltaX = deltaX_high | deltaX_low;              //Combines the low/high bits of X and Y to be one variable.
    deltaY = deltaY_high | deltaY_low;
}


//=========================================================================
void printData(void)
{
    if((deltaX != 0) || (deltaY != 0))      //If there is deltaX or deltaY movement, print the data.
    {
        totalX += deltaX;
        totalY += deltaY;
        
        pc.printf("deltaX: %d\t\t\tdeltaY: %d\n\r", deltaX, deltaY);    //Prints each individual count of deltaX and deltaY.
        pc.printf("X-axis Counts: %d\t\tY-axis Counts: %d\n\r", totalX, totalY);  //Prints the total movement made during runtime.
    }
    
    deltaX = 0;                             //Resets deltaX and Y values to zero, otherwise previous data is stored until overwritten.
    deltaY = 0;
}
