//=========================================================================
//Communication pinouts for serial COM port, SPI, and interrupts
//=========================================================================
static Serial pc(USBTX, USBRX);                     //PC comm
static SPI spi(p23, p24, p25);                      //mosi, miso, sclk
static DigitalOut cs(p22);                          //chip select


//=========================================================================
//Variables and arrays used for communications and data storage
//=========================================================================
int8_t deltaX_low, deltaY_low;                      //Stores the low-bits of movement data.
int16_t deltaX_high, deltaY_high;                   //Stores the high-bits of movement data.
int16_t deltaX, deltaY;                             //Stores the combined value of low and high bits.
int16_t totalX, totalY = 0;                         //Stores the total deltaX and deltaY moved during runtime.
int startupFail;


//=========================================================================
//Functions used to communicate with the sensor and grab/print data
//=========================================================================
uint8_t readRegister(uint8_t addr);
//This function takes an 8-bit address in the form 0x00 and returns an 8-bit value in the form 0x00.

void writeRegister(uint8_t addr, uint8_t data);
//This function takes an 8-bit address and 8-bit data. Writes the given data to the given address.

void startupCheck(void);
//Sets all of the registers needed for initialization and performance optimization.

void initializeSensor(void);
//Sets all of the registers needed for initialization and performance optimization.

void grabData(void);
//Grabs the deltaX and deltaY information from the proper registers and formats it into the proper format.

void printData(void);
//Prints the data out to a serial terminal.





//=========================================================================
//Functions definitions
//=========================================================================
uint8_t readRegister(uint8_t addr)
{
    cs = 0;                                 //Set chip select low/active
    addr = addr & 0x7F;                     //Set MSB to 0 to indicate read operation
    spi.write(addr);                        //Write the given address
    wait_us(35);                             //Add a tiny delay after sending address for some internal cycle timing.
    uint8_t data_read = spi.write(0x00);    //Throw dummy byte after sending address to receieve data
    cs = 1;                                 //Set chip select back to high/inactive
    return data_read;                       //Returns 8-bit data from register
}


//=========================================================================
void writeRegister(uint8_t addr, uint8_t data)
{
    cs = 0;                         //Set chip select low/active
    addr = addr | 0x80;             //Set MSB to 1 to indicate write operation
    spi.write(addr);                //Write the given address
    spi.write(data);                //Write the given data
    cs = 1;                         //Set chip select back to high/inactive
    
    //pc.printf("R:%2X, D:%2X\n\r", addr, readRegister(addr));
            //Uncomment this line for debugging. Prints every register write operation.
}


//=========================================================================
void startupCheck(void)
{
    startupFail = 0;
    writeRegister(0x7F, 0x00);
    writeRegister(0x55, 0x01);
    writeRegister(0x50, 0x07);
    writeRegister(0x7F, 0x0E);
    writeRegister(0x43, 0x10);
    
    for(int i=0; i<3; i++)
    {
        if(readRegister(0x47) != 0x08)
        {
            writeRegister(0x43, 0x10)
            startupFail++;
        }
    }
}


//=========================================================================
void initializeSensor(void)
{
    writeRegister(0x7F, 0x00);
    writeRegister(0x51, 0x7B);
    writeRegister(0x50, 0x00);
    writeRegister(0x55, 0x00);
    writeRegister(0x7F, 0x0E);
    
    if(readRegister(0x73) == 0x00)
    {
        writeRegister(0x7F, 0x00);
        writeRegister(0x61, 0xAD);
        writeRegister(0x51, 0x70);
        writeRegister(0x7F, 0x0E);
        
        if(readRegister(0x70) <= 28)
            writeRegister(0x70, readRegister(0x70) + 14);
            
        else
            writeRegister(0x70, readRegister(0x70) + 11);
            
        writeRegister(0x71, readRegister(0x71) * 45/100);
    }
    
    writeRegister(0x7F, 0x00);
    writeRegister(0x61, 0xAD);
    writeRegister(0x7F, 0x03);
    writeRegister(0x40, 0x00);
    writeRegister(0x7F, 0x05);
    
    writeRegister(0x41, 0xB3);
    writeRegister(0x43, 0xF1);
    writeRegister(0x45, 0x14);
    writeRegister(0x5B, 0x32);
    writeRegister(0x5F, 0x34);
    writeRegister(0x7B, 0x08);
    writeRegister(0x5E, 0x34);
    writeRegister(0x70, 0xE5);
    writeRegister(0x71, 0xE5);
    writeRegister(0x7F, 0x06);
    writeRegister(0x44, 0x1B);
    writeRegister(0x40, 0xBF);
    writeRegister(0x4E, 0x3F);
    writeRegister(0x7F, 0x08);
    writeRegister(0x66, 0x44);
    writeRegister(0x65, 0x20);
    writeRegister(0x6A, 0x3A);
    writeRegister(0x61, 0x01);
    writeRegister(0x62, 0x01);
    writeRegister(0x7F, 0x09);
    writeRegister(0x4F, 0xAF);
    writeRegister(0x5F, 0x40);
    writeRegister(0x48, 0x80);
    writeRegister(0x49, 0x80);
    writeRegister(0x57, 0x77);
    writeRegister(0x60, 0x78);
    writeRegister(0x61, 0x78);
    writeRegister(0x62, 0x08);
    writeRegister(0x63, 0x50);
    writeRegister(0x7F, 0x0A);
    writeRegister(0x45, 0x60);
    writeRegister(0x7F, 0x00);
    writeRegister(0x4D, 0x11);
    writeRegister(0x55, 0x80);
    writeRegister(0x74, 0x1F);
    writeRegister(0x75, 0x1F);
    writeRegister(0x4A, 0x78);
    writeRegister(0x4B, 0x78);
    writeRegister(0x44, 0x08);
    writeRegister(0x45, 0x50);
    writeRegister(0x64, 0xFF);
    writeRegister(0x65, 0x1F);
    writeRegister(0x7F, 0x14);
    writeRegister(0x65, 0x67);
    writeRegister(0x66, 0x08);
    writeRegister(0x63, 0x70);
    writeRegister(0x6F, 0x1C);
    writeRegister(0x7F, 0x15);
    writeRegister(0x48, 0x48);
    writeRegister(0x7F, 0x07);
    writeRegister(0x41, 0x0D);
    writeRegister(0x43, 0x14);
    writeRegister(0x4B, 0x0E);
    writeRegister(0x45, 0x0F);
    writeRegister(0x44, 0x42);
    writeRegister(0x4C, 0x80);
    writeRegister(0x7F, 0x10);
    writeRegister(0x5B, 0x02);
    writeRegister(0x7F, 0x07);
    writeRegister(0x40, 0x41);
    
    wait_ms(10);
    
    writeRegister(0x7F, 0x00);
    writeRegister(0x32, 0x00);
    writeRegister(0x7F, 0x07);
    writeRegister(0x68, 0xF0);
    writeRegister(0x69, 0x00);
    writeRegister(0x7F, 0x0D);
    writeRegister(0x48, 0xC0);
    writeRegister(0x6F, 0xD5);
    writeRegister(0x7F, 0x00);
    writeRegister(0x5B, 0xA0);
    writeRegister(0x4E, 0xA8);
    writeRegister(0x5A, 0x90);
    writeRegister(0x40, 0x80);
    writeRegister(0x7F, 0x08);
    writeRegister(0x6F, 0x9F);
    
    wait_ms(10):
    
    writeRegister(0x6F, 0x00);
    writeRegister(0x7F, 0x14);
    writeRegister(0x6F, 0x1C);
}


//=========================================================================
void grabData(void)
{
    deltaX_low = readRegister(0x03);        //Grabs data from the proper registers.
    deltaX_high = (readRegister(0x04)<<8) & 0xFF00;      //Grabs data and shifts it to make space to be combined with lower bits.
    deltaY_low = readRegister(0x05);
    deltaY_high = (readRegister(0x06)<<8) & 0xFF00;
        
    deltaX = deltaX_high | deltaX_low;      //Combines the low and high bits.
    deltaY = deltaY_high | deltaY_low;
}


//=========================================================================
void printData(void)
{
    if((deltaX != 0) || (deltaY != 0))      //If there is deltaX or deltaY movement, print the data.
    {
        totalX += deltaX;
        totalY += deltaY;
        
        pc.printf("deltaX: %d\t\t\tdeltaY: %d\n\r", deltaX, deltaY);    //Prints each individual count of deltaX and deltaY.
        pc.printf("X-axis Counts: %d\t\tY-axis Counts: %d\n\r", totalX, totalY);  //Prints the total movement made during runtime.
    }
    
    deltaX = 0;                             //Resets deltaX and Y values to zero, otherwise previous data is stored until overwritten.
    deltaY = 0;
}
