#ifndef _FILE_CURRENTMONITOR_H
#define _FILE_CURRENTMONITOR_H

#include "mbed.h"
#include "RTCStore.h"

#define GPREG_AH_COUNTER    0                                // rtc GPREG offset for the coulomb counter
#define GPREG_AH_CAPACITY   1                                // rtc GPREG offset for the capacity spec

const float DEFAULT_AH = 1.5;                                // Default amphours of battery, in case store read is bad/empty
const float DEFAULT_SOC = 0.5;                               // Defualt SOC, in case of bad read/store

const float MIN_CAPACITY_SETTING = 0.5;                      // Lowest allowable capacity setting
const float MAX_CAPACITY_SETTING = 10;                       // Largest allowable capacity setting

enum GLV_BAT_ERROR {
    OVER_CHARGE_I       = 1<<0,     // Charge current exceeded (after filter)
    OVER_DISCHARGE_I    = 1<<1,     // Discharge current exceeded (after filter)
    CAP_INIT            = 1<<2,     // Capacity RTC register was invalid on startup
    SOC_INIT            = 1<<3,     // SOC RTC register was invalid on startup
};

class CoulombCounter
{

public:

    // Configures for a certain pin, millisecond sample period, and which GPREG in store to use to store the ampHours
    CoulombCounter(PinName _pin, int _mSec, unsigned int size=50);
    void setup(float* overChargeCurrent, float* overDischargeCurrent);
    bool size(unsigned int size);
    char readError() {
        return errorPacket;
    }
    void clearError() {
        errorPacket = 0;
    }
    float current() {
        return currentFiltered;    // Last current reading in Amps
    }
    void sample();

    float capacity() {
        return store.read(GPREG_AH_CAPACITY);
    }
    float SOC() {
        return ampHours()/capacity();
    }
    float ampHours() {
        return store.read(GPREG_AH_COUNTER);
    }
    bool changeCapacity(float _ampHours) {
        if (_ampHours < MIN_CAPACITY_SETTING || _ampHours > MAX_CAPACITY_SETTING) return false;
        store.write(_ampHours, GPREG_AH_CAPACITY);
        return true;
    }
    bool resetToSOC(float _SOC) {
        if (_SOC < 0 || _SOC > 1) return false;
        store.write(_SOC*capacity(), GPREG_AH_COUNTER);
        return true;
    }
    bool resetToAh(float _ampHours) {
        if (_ampHours < 0 || _ampHours > capacity()) return false;
        store.write(_ampHours, GPREG_AH_COUNTER);
        return true;
    }

private:
    RTCStore store;                 // Access to the RTC registers (battery-backed)
    unsigned int _size;             // Size of buffer
    void updateAvg();               // Used to get average current and update flags
    int mSec;                       // Integration time
    float currentFiltered;          // Filtered current, last result of updateAvg()
    AnalogIn BatISense;             // Analog input pin
    char errorPacket;               // Errors (over-current)
    float *buffArr;                 // The buffer itself
    unsigned int tracker;           // Position to load next sample
    
    float* overChargeCurrent;       // Charge current limit
    float* overDischargeCurrent;    // Discharge current limit
};
#endif