#include "IOobjects.h"
#include "runTime.h"
#include "outDiagnostics.h"
#include "inCommands.h"

void thread_watchDog(void const* args)
{
    tempData.wdtThreadId = Thread::gettid();
    wdt.kick(WDT_TIME);
    while(1) {
        //Thread::wait(100);
        Thread::signal_wait(0x1F);
        wdt.kick();
    }
}
// Startup code to initialize interfaces and fetch the profile
void init()
{
    char str[100];
    sprintf(str, "\033[2J\033[2J\r\n");     // Clear screen

    // GENERATE RESET TYPE MESSAGE
    bool normalReset = true;
    if (wdt.checkFlag()) {                  // Did a watchdog reset occur?
        wdt.clearFlag();                    // Clear flag
        op->faultCode |= WATCHDOG;          // Record fault
        sprintf(str, "System Mgmt Watchdog Reset\r\n");
        normalReset=false;
    }
    if (LPC_SC->RSID & (1<<3)) {            // Did a brownout reset occur?
        LPC_SC->RSID = (1<<3);              // Clear flag
        op->faultCode |= BROWNOUT;          // Record fault
        sprintf(str, "System Mgmt Brownout Reset\r\n");
        normalReset=false;
    }
    if (normalReset) sprintf(str, "System Mgmt Reset\r\n"); // Normal reset message

    // SET INTERUPT PRIORITIES
    NVIC_SetPriority(TIMER0_IRQn, 0);
    NVIC_SetPriority(PWM1_IRQn, 1);
    NVIC_SetPriority(CAN_IRQn, 2);
    NVIC_SetPriority(UART0_IRQn, 3);
    NVIC_SetPriority(TIMER3_IRQn, 4);

    // SET SYSTEM TIME (RTC)
    int t = time(NULL);                     // Get RTC time
    if (t == 0 || t == -1) set_time(0);     // Seed the timer if it was never setup
    tempData.timeSynced = false;            // Time unsynced on reset, use AMS as master clock
    op->startTime = time(NULL);             // Capture the startup time
    op->SysTime = time(NULL);               // Caputre the system time
    
    // LOAD LAST-USED CONFIGURATION PROFILE
    if (!Profile::loadStartUp()) {
        strncat(str, "Profile load Failed, using default\r\n", 99);   
    }
    op->profileIndex = Profile::usingProfile();
    wdt.kick();                                     // Kick watchdog timer before the wait
    wait(0.5);                                      // Wait so above messages aren't immediately over-written
    
    // ALLOCATE/RESIZE BUFFERS
    glvBat.size(param->glvBat_taps);
    dcdc.size(param->dcdc_taps);
    can.txSize(param->CANtxSize);
    can.rxSize(param->CANrxSize);
    pc.baud(param->SerialBaud);
    pc.txBufferSetSize(param->SerialTxSize);
    //xbeeRelay.baud(param->XbeeBaud);
    //xbeeRelay.txSize(param->XbeeTxSize);
    //xbeeRelay.rxSize(param->XbeeRxSize);
    
    // INITIAL SETUP, LINK PROFILE VARIABLES
    glvBat.changeCapacity(param->nominalCapacity);
    glvBat.setup(&(param->chargeCurrent), &(param->dischargeCurrent));
    dcdc.setup(&(param->dcdcThreshold), &(param->dcdcOverCurrent), &(param->dcdcStartDelay), &(param->dcdcStopDelay));
    AMSlatch.setup(&(param->amsStartDelay));
    IMDlatch.setup(&(param->imdStartDelay));
    AMSlatch.delayStart();
    IMDlatch.delayStart();
    
    // SETUP SERIAL PORT
    pc.format(8, SerialBase::None, 2);      // 2 Stop bits, reduce bad serial packets
    str[99] = 0;
    pc.printf(str);
}

int main()
{
    init();
    
    // Start the watchdog check-in thread
    Thread watchdogCheck(thread_watchDog, 0, osPriorityRealtime, 128);

    // Start the 100Hz data timer (priotity high)
    RtosTimer sample(runTime::thread_sample, osTimerPeriodic);
    sample.start(FAST_LOOP*1000);

    // Start the serial, CAN threads
    Thread serial_out(outDiagnostics::thread_serialOut, 0, osPriorityBelowNormal);
    Thread can_out(outDiagnostics::thread_canOut, 0, osPriorityBelowNormal, 512);

    // Start the input polling thread
    Thread getInputs(inCommands::thread_getInputs, 0, osPriorityLow);

    // Main task
    while(1) {
        runTime::gather(0);
        osSignalSet((osThreadId)(tempData.wdtThreadId), 1<<0);      // Signal watchdog thread
        Thread::wait(GATHER_LOOP*1000);
    }
}