#include "CAN-xbee.h"

// Creates stringified version of CANMessage msg in char* buff (must be at least 15 chars)
// Returns -1 for bad message, or else returns size of char* buff including the terminating '\n'
int convert2array(CANMessage &msg, char* buff) {
    
    // Check message integrity
    if (msg.len > 8) return -1;
    if ((msg.format == CANStandard) && ((msg.id & 0x7FF) != msg.id)) return -1;
    if ((msg.format == CANExtended) && ((msg.id & 0x1FFFFFFF) != msg.id)) return -1;
    
    int i = 0;
    buff[i++] = 'C';      // Start of message
    buff[i++] = (msg.format << 0) | (msg.type << 1) | ((msg.len & 0xf) << 2);     // Header byte, message info on ID size, RTR, # data bytes
    if (msg.format == CANStandard) {
        buff[i++] = (msg.id & 0x0FF);               // Lower byte of ID
        buff[i++] = (msg.id & 0x700) >> 8;          // Upper byte of ID
    } else {
        buff[i++] = (msg.id & 0x000000FF);          // Lowest byte of ID
        buff[i++] = (msg.id & 0x0000FF00) >> 8;
        buff[i++] = (msg.id & 0x00FF0000) >> 16;
        buff[i++] = (msg.id & 0x1F000000) >> 24;    // Highest byte of ID
    }
    for (int j = 0; j < msg.len; j++) {
        buff[i++] = msg.data[j];
    }
    buff[i++] = '\n';
    return i;
}

bool convert2msg(CANMessage &msg, char* buff) {
    // Check for 'D' for done
    if (buff[0] != 'D') return -1;
    
    bool extended = buff[1] & 1;
    bool rtr = buff[1] & 2;
    char DLC = buff[1] >> 2;
    if (DLC > 8) return false;     // Bad length
    
    // Standard frame, get id and data
    if (!extended) {
        if (buff[3] & 0x7 != buff[3]) return false;                 // Last byte of ID bad
        msg.id = buff[2] | (int)(buff[3] << 8);                     // Build the ID
        for (int i = 0; i < DLC; i++) msg.data[i] = buff[i+4];      // Build the data array
        
    // Extended frame, get id and data
    } else {
        if (buff[5] & 0x1F != buff[5]) return false;                            // Last byte of ID bad
        msg.id = buff[2] | (buff[3] << 8) | (buff[4] << 16) | (buff[5] << 24);  // Build the ID
        for (int i = 0; i < DLC; i++) msg.data[i] = buff[i+6];                  // Build the data array
    }
    msg.len = DLC;
    if (rtr) msg.type = CANRemote;
    else msg.type = CANData;
    if (extended) msg.format = CANExtended;
    else msg.format = CANStandard;
    return true;                        // Successfully parsed, passed all checks, arguement was updated
}

CANxbee::CANxbee(PinName tx, PinName rx, int _baud, int txSize, int rxSize) : serial(tx, rx, txSize, rxSize) {
    serial.baud(_baud);
    rx_i = 0;
    rxBuff[0] = 0;
    getChars = -1;
}

// Send a CAN message, first reformat it into a char array, then send over Xbees
bool CANxbee::send(CANMessage &msg, unsigned int* length) {
    char buff[15];                      // Build the string-ified CAN message here
    int size = convert2array(msg, buff);
    *length = 0;
    if (size == -1) return false;       // Bad message, string not formed
    
    int bytesLeft = serial.txBufferGetSize(0) - serial.txBufferGetCount();
    
    if (bytesLeft >= size) {
        for (int i = 0; i < size; i++) serial.putc(buff[i]);    // Send the message out
        *length = size;
        return true;
    }
    return false;
}

// Continuously call this function in main program, when it returns true, it holds a newly received CAN message that came via Xbee
bool CANxbee::receive(CANMessage &msg) {
    int newChar = serial.getcNb();
    if (newChar == -1) return false;        // No new char
    char c = newChar & 0xFF;                // Cast to char
    
    // Listen for a 'C', start of new message as long as not already building message
    if (c == 'C' && (rxBuff[0] != 'C')) {
        rx_i = 0;               // Reset to start of message
        rxBuff[rx_i++] = 'C';   // Add the 'C', increment
        return false;
    }
    
    // 'C' already found, now filling in contents of message
    if (rxBuff[0] == 'C') {
        if (getChars == -1) {           // Get the header byte
            rxBuff[rx_i++] = c;         // Add to string, increment
            bool extended = c & 1;      // Is this an extended message? (4 ID bytes)
            char DLC = c >> 2;          // How long is the data section?
            if (DLC > 8) {              // Bad DLC, reset variables
                getChars = -1;
                rxBuff[0] = 0;
                rx_i = 0;
                return false;
            }
            getChars = DLC + extended?4:2;  // Need to obtain getChars more characters to complete this message
        } else {                            // This is not a header byte, add to contents of message
            rxBuff[rx_i++] = c;
            getChars--;
            if (getChars <= 0) {            // Just added the last char, done message, reset variables
                rxBuff[0] = 'D';            // Mark 'D' for done
                rx_i = 0;
                getChars = -1;
            }
        }
    }
    if (rxBuff[0] != 'D') return false;
    
    // A string is ready, process it here
    if (convert2msg(msg, rxBuff)) return true;
    else return false;
}