#ifndef _Profile_H
#define _Profile_H

// The defines below are used to create change functions
// The change functions allow changing things in the Profile with limits on allowed values
#define CHANGE_FUNC(NAME, LLIM, ULIM)               \
template<class T>                                   \
bool change_##NAME(T data) {                        \
    if (data > ULIM || data < LLIM) return false;   \
    if (data == NAME) return false;                 \
    NAME = data;                                    \
    return true;                                    \
}

// Stores configurable parameters like limits, constants, etc.
// NOTE there are no mutexes/protection mechanisms! Ensure single producer (inCommands.cpp - main thread), single consumer
// To change a member of class Profile, use the change functions defined.  They will enforce limits on the changes.
class Profile
{
public:
    // GLV Battery
    float chargeCurrent;
    float dischargeCurrent;
    float nominalCapacity;
    unsigned int glvBat_taps;
    CHANGE_FUNC(chargeCurrent,    -10, 0)
    CHANGE_FUNC(dischargeCurrent,   0, 10)
    CHANGE_FUNC(nominalCapacity,    0, 10)
    CHANGE_FUNC(glvBat_taps,        1, 1000)

    // DC-DC converter
    float dcdcThreshold;
    float dcdcOverCurrent;
    float dcdcStartDelay;
    float dcdcStopDelay;
    unsigned int dcdc_taps;
    CHANGE_FUNC(dcdcThreshold,      0, 10)
    CHANGE_FUNC(dcdcOverCurrent,    0, 100)
    CHANGE_FUNC(dcdcStartDelay,     0, 10)
    CHANGE_FUNC(dcdcStopDelay,      0, 10)
    CHANGE_FUNC(dcdc_taps,          1, 1000)

    // Latch Circuits
    float imdStartDelay;
    float amsStartDelay;
    CHANGE_FUNC(imdStartDelay,      0, 100)
    CHANGE_FUNC(amsStartDelay,      0, 100)

    // Over-temp
    float internalOverTemp;
    CHANGE_FUNC(internalOverTemp,   0, 100)

    // Boolean states / modes
    bool CANnoAck;                      // Is noAck mode on?
    bool extendedSerial;
    CHANGE_FUNC(CANnoAck,           0, 1)
    CHANGE_FUNC(extendedSerial,     0, 1)

    // Buffer sizes
    unsigned int CANtxSize;             // Size of CAN TX buffer
    unsigned int CANrxSize;             // Size of CAN RX buffer
    unsigned int SerialBaud;            // Serial port baudrate
    unsigned int SerialTxSize;          // Serial TX buffer size
    CHANGE_FUNC(CANtxSize,          1,    1000)
    CHANGE_FUNC(CANrxSize,          1,    1000)
    CHANGE_FUNC(SerialBaud,         9600, 921600)
    CHANGE_FUNC(SerialTxSize,       1,    10000)

    // Load / store / fetch functions
    static bool loadProfile(int index);                     // Load profile from flash into the current RAM object, 0=default, -1=freeze frame
    static bool saveProfile(int index);                     // Save the current RAM profile into flash to index, 0=default
    static bool getProfile(Profile **ptr, int index);       // Retrieve the handle to a profile
    static bool loadStartUp();                              // Load the profile on startup
    static int  usingProfile();                             // Return the last profile loaded into RAM, 0=default, -1=freeze frame
};
class Profile_checkSum
{
public:
    Profile param;
    uint32_t BSDchecksum;
};
#endif
