#include "mbed.h"
#include "rtos.h"
#include "TextLCD.h"
#include <stdio.h>
#include <stdlib.h>

#define RUN 0x1

TextLCD lcd(p15, p16, p17, p18, p19, p20, TextLCD::LCD16x2);
Serial pc (USBTX, USBRX);

// LEDs
DigitalOut leds[] = {LED1, LED2, LED3, LED4}; // 1 = VP, 2 = AP, 3 = AS, 4 = VS

// create log file and debugging vars
//LocalFileSystem local("local");
//FILE *fp = fopen("/local/out.txt", "w");
int between_a = 0;
int between_v = 0;

// Heart Signals
int AG = 0;
int VG = 0;

// Normal Values
const int N_PVARP = 325;    // ms
const int N_VRP   = 300;    // ms
const int N_LRI   = 857;    // ms (= about 70ppm)
const int N_AVI   = 65;     // ms
const int N_UB    = 100;    // 100ppm
const int N_LB    = 40;     // 40ppm

// Heart Values - Normal Mode is default
int PVARP = N_PVARP;
int VRP = N_VRP;
int LRI = N_LRI;
int default_LRI = N_LRI;
int AVI = N_AVI;
int UB = N_UB;
int LB = N_LB;

// time vars
Timer global_t;
int isVRP = 0;
int isPVARP = 0;
int waitingForV = 1;

// functions
void VP_func(void const *args);
void AP_func(void const *args);
void VS_func(void const *args);
void AS_func(void const *args);
void PM_monitor_func(void const *args);
void manage_signals(void const *i);
void flashLED(int i);
void event_out(char *s, int between_t);
void blind();
//void rand_heart_func(void const *args);

// threads
Thread * VS_thread;
Thread * AS_thread;
//Thread * rand_heart_thread; // just for testing until mbed connection is made

// rtos timers
RtosTimer * VP_timer;
RtosTimer * AP_timer;
RtosTimer * VRP_timer;
RtosTimer * PVARP_timer;
//RtosTimer * exit_timer; // for log file

int main() {
    
    // start global timer
    global_t.start();
    
    // init threads
    VS_thread = new Thread(VS_func);
    AS_thread = new Thread(AS_func);
    //rand_heart_thread = new Thread(rand_heart_func);    // just for testing until mbed connection is made
    
    // init timers
    VP_timer = new RtosTimer(VP_func, osTimerOnce, (void *)0);
    AP_timer = new RtosTimer(AP_func, osTimerOnce, (void *)0);
    VRP_timer = new RtosTimer(manage_signals, osTimerOnce, (void *)1);
    PVARP_timer = new RtosTimer(manage_signals, osTimerOnce, (void *)2);
    //exit_timer = new RtosTimer(manage_signals, osTimerOnce, (void *)99);
    
    // init global time, VP thread, exit thread
    VP_timer->start(AVI);
    //exit_timer->start(60000);
    
    // main thread
    while (1) {
        
    }
}

void manage_signals(void const *i) {
    if ((int)i==1) isVRP = 0;
    if ((int)i==2) isPVARP = 0;
    
    // for debuggging
    //if ((int)i==99) {
    //fclose(fp);
    //exit(1);
    //}
}

void AP_func(void const *args) {
    
    // start VP timer
    VP_timer->start(AVI);
    
    // update state
    waitingForV = 1;
    
    // output
    event_out("AP",between_a);
    between_a = global_t.read_ms();
    
    // flash LED
    flashLED(2);
}

void VP_func(void const *args) {
    
    // start AP timer
    AP_timer->start(LRI-AVI);
    
    // update state
    waitingForV = 0;
    
    // set VRP, PVARP
    blind();
    
    // output
    event_out("VP", between_v);
    between_v = global_t.read_ms();
    
    // flash LED
    flashLED(1);
}

void AS_func(void const *args) {
    while (1) {
        
        // wait for event
        Thread::signal_wait(RUN,osWaitForever);
        
        // update state
        waitingForV = 1;
        
        // stop AP timer and start VP timer
        AP_timer->stop();
        VP_timer->start(AVI);
        
        // output
        event_out("AS", between_a);
        between_a = global_t.read_ms();
        
        // flash LED
        flashLED(3);
    }
}

void VS_func(void const *args) {
    while (1) {
        
        // wait for event
        Thread::signal_wait(RUN,osWaitForever);
        
        // update state
        waitingForV = 0;
        
        // stop VP timer and start AP timer
        VP_timer->stop();
        AP_timer->start(LRI-AVI);
        
        // set VRP, PVARP
        blind();
        
        // output
        event_out("VS", between_v);
        between_v = global_t.read_ms();
        
        // flash LED
        flashLED(4);
    }
}

/*
 void rand_heart_func(void const *args) {
 int interval;
 srand(time(NULL));
 while (1) {
 interval = rand()%5000+10;
 fprintf(fp,"interval = %d\n",interval);
 Thread::wait(interval);
 if (interval%2) AG = 1;
 else
 VG = 1;
 }
 }
 */

void PM_monitor_func(void const *args) {
    while (1) {
        if (AG == 1) {
            //fprintf(fp,"%f\tAget\t%d\n",global_t.read_ms(),isPVARP);
            AG = 0;
            if (!isPVARP && !waitingForV) AS_thread->signal_set(RUN);
        }
        if (VG == 1) {
            //fprintf(fp,"%f\tVget\t%d\n",global_t.read_ms(),isVRP);
            VG = 0;
            if (!isVRP && waitingForV) VS_thread->signal_set(RUN);
        }
    }
}

void flashLED(int i) {
    leds[i-1] = 1;
    wait(0.01);
    leds[i-1] = 0;
}

void event_out(char *s, int between_t) {
    lcd.printf("%d\t%s\t%d\n",global_t.read_ms(),s,global_t.read_ms()-between_t);
    //fprintf(fp, "%f\t%s\t%f\n",global_t.read_ms(),s,global_t.read_ms()-between_t);
}

void blind() {
    isVRP = 1;
    isPVARP = 1;
    VRP_timer->start(VRP);
    PVARP_timer->start(PVARP);
}
