/* ********************************************
* PCD8544 Driver for an LCD with an PCD8544
* controller.
* Note: Text mode supported only.
*
* Created on: 12/31/2010 at 19:48
* Created by: CarlosFTM
********************************************** */

#include "mbed.h"
#include "pcd8544_drv.hpp"

pcd8544::pcd8544(PinName pin_sclk, PinName pin_mosi, PinName pin_dc, PinName pin_sce, PinName pin_reset) :  _sclk(pin_sclk),
                                                                                                            _mosi(pin_mosi),
                                                                                                            _dc(pin_dc),
                                                                                                            _sce(pin_sce),
                                                                                                            _reset(pin_reset)
{
  /* Set all pins to initial state */
  _sce   = HIGH;
  _reset = HIGH;
  _sclk  = LOW;
}

/* Transmits a single byte to the LCD */
void pcd8544::sendByte(char byte, bool command)
{
  int loop;
  _dc  = !command;
  _sce = LOW;
  for (loop = 0; loop < 8; loop++)
  {
    _mosi = ((byte << loop) & 0x80); // TX starts from MSB
    clockTick(TICK);
  }
  _sce = HIGH;
}

/* Writes a data byte into LCD RAM*/
void pcd8544::writeData(char data)
{
  sendByte(data, false);
}

/* Sends a command to the LCD */
void pcd8544::writeCmd(char cmd)
{
  sendByte(cmd, true);
}

/* Writes a ASCII character into LCD RAM */
void pcd8544::writeChar(char character)
{
  char loop = 0;

  if ((character >= 32) && (character <= 122))
  {
    character = character - 32;
  }
  else
  {
    character = 0x00;
  }

  for (loop = 0; loop < 10; loop++)
  {
    sendByte(ascii_table[character][loop], false);
  }
  sendByte(0x00, false);
}

/* Writes a string of characters into LCD RAM */
void pcd8544::writeString(char* character)
{
  while (*character)
  {
    writeChar(char(*character));
    character++;
  }
}

/* Resets the LCD*/
void pcd8544::resetLCD()
{
  int loop;
  _sce   = LOW;
  wait_us(TICK);
  _reset = LOW;
  for (loop = 0; loop < RESET_TICKS; loop++)
  {
    clockTick(TICK);
  }
  _reset = HIGH;
  wait_us(TICK);
  _sce = HIGH;
}

/* Clear the LCD RAM*/
void pcd8544::clearLCD()
{
  writeCmd(0x80); // Set X ram-address to 0x00
  writeCmd(0x40); // Set Y ram-address to 0x00

  int loop = 0;
  
  for (loop = 0; loop < (MAX_PIX_X * CHAR_PIX_X); loop++)
  {
    writeData(0x00); // Write empty character
  }
  /* Set cursor to position 0,0 */
  writeCmd(0x80); // Set X ram-address to 0x00
  writeCmd(0x40); // Set Y ram-address to 0x00
}

/* Initialize the LCD with default values */
void pcd8544::initLCD(void)
{
  writeCmd(0x90);  // Set Vop as in PCD8544 specs example
  writeCmd(0x20);  // Normal instruction set. Horizontal addressing.
  writeCmd(0x0C);  // display in normal mode

  /* Set cursor to position 0,0 */
  writeCmd(0x80); // Set X ram-address to 0x00
  writeCmd(0x40); // Set Y ram-address to 0x00
}


/* Sends a clock tick to the LCD */
inline void pcd8544::clockTick(unsigned short useg)
{  
    _sclk = LOW;
    wait_us(useg);
    _sclk = HIGH;
    wait_us(useg);
}

/* Set the cursor position on specific character position x,y*/
void pcd8544::setCursorXY(char x, char y)
{
  x = (x | SET_ADDRES_X);
  y = (y  | SET_ADDRES_Y);
  writeCmd(x); // Set X ram-address to x
  writeCmd(y); // Set Y ram-address to y
}