#include "Sampling.h"
#include "TimeInterface.h"
#include "mbed.h"
#include "rtos.h"

//Thread Sync Tools
Mutex tempReadingsLock;
Mutex presReadingsLock;
Mutex LDRReadingsLock;
Mutex timeReadingsLock;

//Buffers with zero inital value
float tempReadings[BUFFERSIZE] = {};
float presReadings[BUFFERSIZE] = {};
float LDRReadings[BUFFERSIZE] = {};
time_t timeReadings[BUFFERSIZE] = {};

unsigned short SAMPLERATE = 15; //Set initial sample rate.

Thread t1; //Sample Enviromental Sensor
Thread t2; //Sample LDR Sensor

Ticker sampleRate;          
Timeout SampleLEDTimeout;   

bool NewEnvSample;  //Is there new data from the envirom sensor to output?
bool NewLDRSample;  //Is there new data from the LDR to output?

//Index
volatile unsigned short nextIndex = 0;
volatile unsigned short currentIndex = 0;
volatile unsigned short oldestIndex = 0;

bool firstSample;

void SampleTimerISR(void)
{
    //Flag Threads
    t1.signal_set(1);
    t2.signal_set(1);
    SamplingLED = 1;
    SampleLEDTimeout.attach(&FlipSamplingLED,0.1); //To turn LED off
}

void ConfigThreadsAndIR(void)
{
    NewEnvSample = false;  //Reset
    NewLDRSample = false;  //Reset

    t1.start(&ThreadSampleEnvSensor); //Start Threads
    t2.start(&ThreadSampleLDR);

    Thread t1(osPriorityRealtime);  //Higher priority
    Thread t2(osPriorityRealtime);

    sampleRate.attach(&SampleTimerISR, SAMPLERATE); //15 second interval
}

void AddTempSample(float temp)
{
    tempReadingsLock.lock();        //Take the key
    tempReadings[nextIndex] = temp; //Add the sample after the most recent
    tempReadingsLock.unlock();      // Release the key
}

void AddPresSample(float pres)
{
    presReadingsLock.lock();        //Take the key
    presReadings[nextIndex] = pres; //Add to register
    presReadingsLock.unlock();      //Release the key
}

void ThreadSampleEnvSensor(void)
{
    while (true) {
        Thread::signal_wait(1); //Wait for signal 1
        //Get readings
        float temp = sensor.getTemperature();
        float pres = sensor.getPressure();
        AddPresSample(pres);    //Add value to register
        AddTempSample(temp);    //Add value to register
        NewEnvSample = true;    //Signal to main thread
    }
}

void AddLDRSample(float LDRval)
{
    LDRReadingsLock.lock();             //Take the key
    LDRReadings[nextIndex] = LDRval;    //Add the sample after the most recent
    LDRReadingsLock.unlock();           // Release the key
}

void AddTimeSample(time_t sampledTime)
{
    timeReadingsLock.lock();                //Take the key
    timeReadings[nextIndex] = sampledTime;  //Add the sample after the most recent
    timeReadingsLock.unlock();              // Release the key
}

void ThreadSampleLDR(void)
{
    while (true) {
        Thread::signal_wait(1); //Wait for signal 1
        //get readings
        float LDRval = LDRSensor; //Read the analogue pin value
        time_t currentTime = time(0);   //Get the system time
        AddLDRSample(LDRval);

        AddTimeSample(currentTime);
        NewLDRSample = true;    //signal to main thread
    }
}

void IncrementIndex(void)
{
    nextIndex = IndexIncrement(nextIndex); //Increment next index
    if (firstSample) {
        firstSample = false; //During first sample, do not increment current or oldest
    } else {
        currentIndex = IndexIncrement(currentIndex);
        if (currentIndex == oldestIndex) { //When current index overflows, start infrementing oldest
            oldestIndex = IndexIncrement(oldestIndex);
        }
    }
}

unsigned short IndexIncrement(unsigned short thisIndex)
{
    if (thisIndex+1 == BUFFERSIZE) {
        thisIndex = 0; //When index reached buffersize, reset to 0
    } else {
        thisIndex++;   //Else increment
    }
    return thisIndex;
}

unsigned short IndexDecrement(unsigned short thisIndex)
{
    if (thisIndex-1 == -1) { // Wait for underflow
        thisIndex = BUFFERSIZE-1; //When index reaches 0 reset to Buffersize-1
    } else {
        thisIndex--;   //Else decrement
    }
    return thisIndex;
}

void Sampling(bool inputState)
{
    if (inputState) {
        sampleRate.attach(&SampleTimerISR, SAMPLERATE); //Attach ticker
    } else {
        sampleRate.detach(); //Detach ticker, stops sampling
    }
}

void TakeKeys(bool inputState)
{
    if (inputState) {
        tempReadingsLock.lock();             //Take the key
        presReadingsLock.lock();
        LDRReadingsLock.lock();
        timeReadingsLock.lock();
    } else {
        tempReadingsLock.unlock();           //Release the key
        presReadingsLock.unlock();
        LDRReadingsLock.unlock();
        timeReadingsLock.unlock();
    }
}


void FlipSamplingLED(void)
{
    SamplingLED = 0;
}

