#ifndef __Sampling__
#define __Sampling__
/*
* This module relates to the handling and sampling of enviromental data.
* Sampled data is held in 4 arrays. One for each different reading. Each
* of these is protected by a mutex lock, which is taken when data is being
* written to the registers.
* There are common indexes for the positions in the array. Stored is the position
* position of the oldest sample, the most recent sample, and where the next
* sample is to be written to. These operate in such a way that the array will fill
* and when fill will start to overwrite the oldest sample.
* Producer functions write samples to nextIndex position, allowing the most
* recent complete sample to continue to be read by consumers.
* 
* A ticker runs at the sample rate interval. This signals two sampling threads.
* Two threads were used to prevent against delay from I2C interface.
* These threads take the samples then write them to the nextIndex, then signal done.
* On seeing that both threads are complete, the main thread then increments the
* indexes and passes the latest variables to the LCD.
*/
#include "mbed.h"
#include "BMP280.h"
#include "rtos.h"

#define BUFFERSIZE 120
extern unsigned short SAMPLERATE;

//Thread Sync Tools
extern Mutex tempReadingsLock;
extern Mutex presReadingsLock;
extern Mutex LDRReadingsLock;
extern Mutex timeReadingsLock;

//Buffers
extern float tempReadings[BUFFERSIZE];
extern float presReadings[BUFFERSIZE];
extern float LDRReadings[BUFFERSIZE];
extern time_t timeReadings[BUFFERSIZE];

extern volatile unsigned short currentIndex;
//Position in the buffer of the newest sample
extern volatile unsigned short nextIndex;
//Position in the buffer where the next sample needs to be writen to
extern volatile unsigned short oldestIndex;
//Position in the buffer of the oldest sample

extern bool firstSample;
//Used to indicate this is the first sample to be taken. used in IncrementIndex func

extern Thread t1; //Sample Enviromental Sensor
extern Thread t2; //Sample LDR Sensor

extern Ticker sampleRate;
extern Timeout SampleLEDTimeout;
//Hardware
extern BMP280 sensor;
extern AnalogIn LDRSensor; //Input pin for LDR
extern DigitalOut SamplingLED; //Onboard LED showing when a sample happens

extern bool NewEnvSample;  //Is there new data from the envirom sensor to output?
extern bool NewLDRSample;  //Is there new data from the LDR to output?

void SampleTimerISR(void);
//Called by ticker. Calls the sample funcs of each device by flagging the threads

void ConfigThreadsAndIR(void);
//Setup Interrupts and Threads

void ThreadSampleEnvSensor(void);
//when flagged by interrupt will capture a sample then calls the addtobufferfuncs

void AddTempSample(float temp);
//Producer function

void AddPresSample(float pres);
//Producer Function

void ThreadSampleLDR(void);
//When flagged by interrupt will read time and LDR value.

void AddLDRSample(float LDR);
//Producer Function

void AddTimeSample(time_t sampledTime);
//Producer Function

void IncrementIndex(void);
//Called after samples are added. Increments the index and moves the oldest
//along if necessary

void FlipSamplingLED(void);
//Called by timeout, turns of LED

unsigned short IndexIncrement(unsigned short thisIndex);
//Incrementing the index with respect for the buffersize. Used to prevent overflow.

unsigned short IndexDecrement(unsigned short thisIndex);
//Decrementing the index with respect for the buffersize. Used to prevent overflow.

void Sampling(bool inputState);
//Start or stop sampling. true = start, false = stop. 

void TakeKeys(bool inputState);
//Lock or unlock sampling variables. true = take keys. false = return keys

#endif