#include "mbed.h"
#include "ADXL375_i2c.h"
#include "LPS25H_lib.h"

#define TIME_PRINT 0.1
#define ALT_SEPARATE 6.0

/*******************************************************************************
コンストラクタ
*******************************************************************************/
Serial pc(USBTX, USBRX, 115200);
Serial sd(p13, p14, 115200);

I2C i2c_bus(p9, p10);

LPS25H_lib LPS25H(LPS25H_lib::AD0_HIGH, i2c_bus);
ADXL375_i2c ADXL375(i2c_bus, ADXL375_i2c::ALT_ADDRESS_HIGH);

Timer timer_rec;
Timer timer_button;
Ticker tick_print;

InterruptIn pinReset(p17);
InterruptIn pinRec(p16);

DigitalOut LED_REC(LED1);
DigitalOut LED_PT(LED2);

/*******************************************************************************
関数のプロトタイプ宣言
*******************************************************************************/
void setup();

void readSensor();
void resetPT();

void printData();
void readPC();

void recData();
void startRec();
void stopRec();
void buttonPush();
void buttonRelease();

/*******************************************************************************
変数の宣言
*******************************************************************************/
int time_rec;

float acc[3];
float pres, temp, alt;
float pres_0, temp_0;
float alt_old, speed;
int separate_c;

bool save = false;
bool separate = false;
bool button_push = false;

/*******************************************************************************
メイン関数
*******************************************************************************/
int main() {
    setup();
    
    pc.attach(&readPC, Serial::RxIrq);
    tick_print.attach(&printData, TIME_PRINT);
    
    pinReset.mode(PullUp);
    pinReset.fall(&resetPT);
    
    pinRec.mode(PullUp);
    pinRec.fall(&buttonPush);
    pinRec.rise(&buttonRelease);
    
    while(1) {
        readSensor();
        recData();
        wait(0.003f);
    }
}

/*******************************************************************************
センサー読み取り
*******************************************************************************/
void readSensor(){
    alt_old = alt;
    pres = LPS25H.getPres();
    temp = LPS25H.getTemp();
    alt = LPS25H.getAlt(pres_0, temp_0);
    speed = (alt - alt_old) / 0.1;
    
    if(save && alt <= ALT_SEPARATE && speed < 0.0f && !separate){
        separate_c ++;
        if(separate_c > 10){
            separate_c = 0;
            separate = true;
        }
    }
    
    ADXL375.getOutput(acc);
}

/*******************************************************************************
高度リセット
*******************************************************************************/
void resetPT(){
    pres_0 = pres;
    temp_0 = temp;
    
    LED_PT = 1;
    wait(0.05f);
    LED_PT = 0;
    wait(0.05f);
    LED_PT = 1;
    wait(0.05f);
    LED_PT = 0;
    wait(0.05f);
    LED_PT = 1;
    wait(0.05f);
    LED_PT = 0;
}

/*******************************************************************************
データをPCで表示
*******************************************************************************/
void printData(){
    pc.printf("%.4f[hPa]\t%.2f[degC]\t%.2f[m]  \t", pres, temp, alt);
    pc.printf("%.2f, %.2f, %.2f\t", acc[0], acc[1], acc[2]);
    pc.printf("%d\r\n", separate);
}

/*******************************************************************************
PCからのコマンド読み取り
*******************************************************************************/
void readPC(){
    char c = pc.getc();
    switch(c){
        case '1':
        startRec();
        break;
        
        case '0':
        stopRec();
        break;
        
        case 'R':
        resetPT();
        break;
    }
}

/*******************************************************************************
データを記録
*******************************************************************************/
void recData(){
    if(save){
        time_rec = timer_rec.read_us();
        sd.printf("%d,", time_rec);
        sd.printf("%f,%f,%f,", pres, temp, alt);
        sd.printf("%.2f,%.2f,%.2f,", acc[0], acc[1], acc[2]);
        sd.printf("%d\r\n", separate);
    }
}

/*******************************************************************************
記録開始
*******************************************************************************/
void startRec(){
    if(!save){
        tick_print.detach();
        
        timer_rec.reset();
        timer_rec.start();
        save = true;
        LED_REC = 1;
    }
}

/*******************************************************************************
記録停止
*******************************************************************************/
void stopRec(){
    if(save){
        save = false;
        LED_REC = 0;
        timer_rec.stop();
        sd.printf("\r\n\n");
        
        tick_print.attach(&printData, TIME_PRINT);
    }
}

/*******************************************************************************
ボタンが押されたとき
*******************************************************************************/
void buttonPush(){
    if(!button_push){
        timer_button.start();
        button_push = true;
    }
}

/*******************************************************************************
ボタンが離されたとき
*******************************************************************************/
void buttonRelease(){
    int time_button;
    
    if(button_push){
        button_push = false;
        time_button = timer_button.read();
        timer_button.stop();
        timer_button.reset();
        if(time_button >= 3){
            if(!save){
                startRec();
            }
            else{
                stopRec();
            }
        }
    }
}

/*******************************************************************************
セットアップ（最初の１回実行）
*******************************************************************************/
void setup(){
    wait(0.5f);
    
    LPS25H.begin(25);
    LPS25H.setFIFO(16);
    if(LPS25H.whoAmI() == 0){
        pc.printf("LPS25H  : OK\r\n");
        readSensor();
        resetPT();
    }
    else{
        pc.printf("LPS25H  : NG.....\r\n");
    }
    
    ADXL375.setDataRate(ADXL375_3200HZ);
    if(ADXL375.whoAmI() == 1){
        pc.printf("ADXL375 : OK\r\n");
    }
    else{
        pc.printf("ADXL375 : NG.....\r\n");
    }
    ADXL375.offset(-0.3f, -0.6f, 0.3f);
    wait(0.5f);
}
