#include "LPS25H_spi.h"
#include "mbed.h"
#include "math.h"

/**********
コンストラクタ
・使用するSPIを選択
・使用するCSピンを選択
**********/
LPS25H_spi :: LPS25H_spi(SPI &userSPI, PinName userPin) : cs(userPin){
    spi = &userSPI;
    
    spi -> format(8, 3);
    spi -> frequency(5000000);
    cs = 0;
    spi -> write(0x00);
    spi -> write(0x00);
    cs = 1;
}
    
/**********
(void)センサー起動(int データレート)
・データ更新レートを選択（1, 10, 25, 50, 75）
**********/
void LPS25H_spi :: begin(int drate){
    switch(drate){
        case 1:
        cmd_ctrl_reg1 = 0x90;
        break;
        
        case 7:
        cmd_ctrl_reg1 = 0xA0;
        break;
        
        case 12:
        cmd_ctrl_reg1 = 0xB0;
        break;
        
        case 25:
        cmd_ctrl_reg1 = 0xC0;
        break;
        
        default:
        cmd_ctrl_reg1 = 0x80;
        break;
    }
    cs = 0;
    spi -> write(CTRL_REG1 & 0x3F);
    spi -> write(cmd_ctrl_reg1);
    cs = 1;
    
    wait(0.1f);
}

/**********
(void)FIFO設定（int FIFOレベル）
・FIFOウォーターマークレベルを選択（2, 4, 8, 16, 32）
**********/
void LPS25H_spi :: setFIFO(int size){
    cmd_ctrl_reg2 = 0x40;
    switch(size){
        case 2:
        cmd_fifo_ctrl = 0xC1;
        break;
        
        case 4:
        cmd_fifo_ctrl = 0xC3;
        break;
        
        case 8:
        cmd_fifo_ctrl = 0xC7;
        break;
        
        case 16:
        cmd_fifo_ctrl = 0xCF;
        break;
        
        case 32:
        cmd_fifo_ctrl = 0xDF;
        break;
        
        default:
        cmd_ctrl_reg2 = 0x00;
        cmd_fifo_ctrl = 0x00;
    }
    
    cs = 0;
    spi -> write(CTRL_REG2 & 0x3F);
    spi -> write(cmd_ctrl_reg2);
    cs = 1;
    cs = 0;
    spi -> write(FIFO_CTRL & 0x3F);
    spi -> write(cmd_fifo_ctrl);
    cs = 1;
    
    wait(0.1f);
}
 
/**********
(int)Who Am I
接続できているかを判定
接続OK：1
接続NG：-1
**********/
int LPS25H_spi :: whoAmI(){
    cs = 0;
    spi -> write(WHO_AM_I | 0x80 & 0xBF);
    ans = spi -> write(0x00);
    cs = 1;
    if(ans == 0xBD){
        return 1;
    }
    else{
        return -1;
    }
}

/**********
(float)気圧を取得
**********/
float LPS25H_spi :: getPres(){
    cs = 0;
    spi -> write(P_XL | 0xC0);
    pres_raw[0] = spi -> write(0x00); //XL
    pres_raw[1] = spi -> write(0x00); //L
    pres_raw[2] = spi -> write(0x00); //H
    cs = 1;
    
    return (float)(pres_raw[0] | pres_raw[1] << 8 | pres_raw[2] << 16) / 4096.0f;
}

/**********
（float）温度を取得
**********/
float LPS25H_spi :: getTemp(){
    cs = 0;
    spi -> write(T_L | 0xC0);
    temp_raw[0] = spi -> write(0x00); //L
    temp_raw[1] = spi -> write(0x00); //H
    cs = 1;
    
    if(temp_raw[1] >= 128){
        return 42.5 - (float)(65536 - (temp_raw[0] | temp_raw[1] << 8)) / 480.0f;
    }
    else{
        return 42.5 + (float)(temp_raw[0] | temp_raw[1] << 8) / 480.0f;
    }
}

/**********
(float)高度を取得(float 0mでの気圧, float 0mでの温度)
・引数に0m地点での気圧・温度を入れる
・計算式のソース：http://www.geocities.jp/u4ren6/Main/Excel_Data0017.html
**********/
float LPS25H_spi :: getAlt(float P_0, float T_0){
    pres_0 = P_0;
    temp_0 = T_0;
    pres_now = getPres();
    
    return -(273.0 + temp_0) / 0.0342 * log(pres_now / pres_0);
}