#ifndef _IM920_H_
#define _IM920_H_
#include "mbed.h"

#define IM920_BUFF_SIZE 160

const int FUNC_NUMBER = 256;


typedef enum im920_buff_classify{
    BUFF_A = 0,
    BUFF_B = 1
}IM920_Buff_Classify;

//解析方法に関する変数
const int WAIT_TIME_US = 1000;
typedef enum{
    DATA_STRING = 0,        //データの文字列
    RESPONSE_STRING = 1,    //OKかNGのレスポンスの文字列
    NORMAL_STRING = 2,
    OTHER_STRING = 3,
    RECEIVE_ID = 4,
    INTERACTIVE_STRING = 5
}IM920_Analyze_Selector;



static IM920_Buff_Classify AB = BUFF_A;

class IM920{
    public:
        IM920(Serial &serial, Serial &pc, int baudrate = 19200);
    private:  
        Serial *ser;
        Serial *_pc;
        
        //バッファ関連の変数
        char buff_A[IM920_BUFF_SIZE];
        char buff_B[IM920_BUFF_SIZE];
        char* readBuffAddr;
        char* writeBuffAddr;
         
        
        char im920_dataLength;
        char im920_sendBuff[129];
        
        int nodeNumber;
        //インタラクティブモードから復帰できるかを示すフラグ
        //1ならインタラクティブモード、0なら通常モード
        int interactiveFlag;
        
        IM920_Analyze_Selector AS;
        
    public:
    
        union IM920_CAST{
            double d;
            float f;
            long l;
            int i;
            short s;
            char c;
            char u[8];
        };
        union IM920_CAST im920_cast; 
        
        char data[64];
        int sendDataSize;
        void (*p_callFunc[256])(void);
        char node;
        char RSSI;//受信強度
        int rID;//受信した送信機ID
        int ID; //自分の固有ID
        
        /**
          @note 無線機の設定を変更するわけではない
          @bref IM920XSとの通信ボーレート変更
          @param baudrate 1200,2400,4800,9600,19200,38400,57600 or 115200
        */
        void baud(int baudrate){
            ser->baud(baudrate);   
        }
        
        /**
          @bref インタラクティブモードへ移行する,@で復帰
        */
        void interactiveMode();
        
        /**
          @bref 16進数文字列データから数値データを取得します
        */
        void data_analyze();
        
        /**
          @bref データ処理関数を設定
          @param *funcAddr 関数ポインタ、void型引数無し関数のみ設定可
          @param header 関数を選択する0~255のヘッダー
        */
        void attach(void (*funcAddr)(void), unsigned header);
    
        /**
          @bref パラメータ不揮発メモリ書き込み許可モードへ移行
        */
        bool enableWrite();
        
            /**
              @bref パラメータ不揮発メモリ書き込み不許可モードへ移行
            */     
        bool disableWrite();
        
            /**
              @bref 固有IDを読み出す
            */
        int readID();
        
        /**
          @bref ノード番号をセットする
        */
        bool setNodeNumber(char nodeNum);   
        
        /**
          @bref ノード番号を読み出す
        */
        int readNodeNumber();   
        
        /**
            @bref 受信する無線機IDを一つ登録する
        */
        bool setReceiveID(int ID);
        
        /**
          @bref 登録されている受信IDを表示する
        */
        void readReceiveID();
        
        /**
          @bref 登録されている受信IDを全消去する
        */
        bool eraseReceiveID();      
        
        /**
          @bref 使用する周波数chを設定する
        */
        bool setChannel(char ch);   
        
        /**
          @bref 使用する周波数chを読み出す
        */
        char readChannel();         
        
        /**
          @bref アスキーコードの文字列で送受信ができるようにする
        */
        bool enableCharacterIO();   
        
        /**
          @bref バイナリ送受信モードへ移行
        */
        bool disableCharacterIO();  
        
        /**
          @bref 送信バッファにdouble型を一つ入れる
        */
        void write(double val);
        
        /**
          @bref 送信バッファにlong型を一つ入れる
        */
        void write(long val);
        
        /**
          @bref 送信バッファにfloat型を一つ入れる
        */
        void write(float val);
        /**
          @bref 送信バッファにint型を一つ入れる
        */
        void write(int val);
        
        /**
          @bref 送信バッファにshort型を一つ入れる
        */
        void write(short val);
        
        /**
          @bref 送信バッファにchar型を一つ入れる
        */
        void write(char val);
        
        /**
          @bref 送信バッファにfloat型を複数入れる
        */
       // void write(float *array, int count);
        
        template <typename T> void write(T *array, int count){
            for(int i = 0;i < count; i++){
                write(array[i]);   
            }
        }
        /**
          @bref 送信バッファを送信する
        */
        void send();
        
        /**
          @bref 送信バッファにヘッダーを付ける、writeする前に使う
        */
        void header(char headerNumber){
            write(headerNumber);
        }
        
        /**
          @bref デバッグ用、受信バッファを表示
        */
        void im920_debug();
        
        /**
          @bref インタラクティブモードでコマンドを表示する
        */
        void printFormat();
        
        /**
          @bref データ配列をshort型に変換する
          @param array データのchar型配列、データの始まるアドレスを入れる
        */
        short toShort(char *array);
        short toShort(int i){
            return toShort(&data[i]);   
        }
        
        /**
          @bref データ配列をint型に変換する
          @param array データのchar型配列、データの始まるアドレスを入れる
        */
        int toInt(char *array);
        int toInt(int i){
            return toInt(&data[i]);   
        }
        /**
          @bref データ配列をfloat型に変換する
          @param array データのchar型配列、データの始まるアドレスを入れる
        */
        float toFloat(char *array);
        float toFloat(int i){
            return toFloat(&data[i]);   
        }
        /**
          @bref データ配列をlong型に変換する
          @param array データのchar型配列、データの始まるアドレスを入れる
        */
        long toLong(char *array);
        long toLong(int i){
            return toLong(&data[i]);   
        }
        
        /**
          @bref データ配列をdouble型に変換する
          @param array データのchar型配列、データの始まるアドレスを入れる
        */
        double toDouble(char *array);
        double toDouble(int i){
            return toDouble(&data[i]);   
        }
        /**
          @bref チェックサムを計算する
          @param count データの個数、何byteか
        */
        char im920CheckSum(int count);
        
        /**
          @bref 受信データのチェックサムの計算結果を入れる
        */
        char receiveCheckSum;
     
    private:
        
        /**
          @bref 1文字受信するごとに呼ばれ、受信バッファに入れる
        */
        void im920Handler();
        
        char checkSum;

        /**
          @bref 何もしない関数
        */
        static void dummyFunc(void);
        
        /**
          @bref 16進数文字を数値に変換する関数 
        */
        int asciiToNumber(char c);
};

inline void IM920::write(double val){
    if(sendDataSize <= 56){
        im920_cast.d = val;
        sprintf(im920_sendBuff, "%s%02X%02X%02X%02X%02X%02X%02X%02X", im920_sendBuff, im920_cast.u[0], im920_cast.u[1], im920_cast.u[2]
                                , im920_cast.u[3], im920_cast.u[4], im920_cast.u[5], im920_cast.u[6], im920_cast.u[7]);
        sendDataSize += 8;
        checkSum = checkSum ^ im920_cast.u[7] ^ im920_cast.u[6] ^ im920_cast.u[5] ^ im920_cast.u[4] ^ im920_cast.u[3] ^ im920_cast.u[2] ^ im920_cast.u[1] ^ im920_cast.u[0];
    }
}

inline void IM920::write(long val){
    if(sendDataSize <= 56){
        im920_cast.l = val;
        sprintf(im920_sendBuff, "%s%02X%02X%02X%02X%02X%02X%02X%02X", im920_sendBuff, im920_cast.u[0], im920_cast.u[1], im920_cast.u[2]
                                , im920_cast.u[3], im920_cast.u[4], im920_cast.u[5], im920_cast.u[6], im920_cast.u[7]);   
        sendDataSize += 8;
        checkSum = checkSum ^ im920_cast.u[7] ^ im920_cast.u[6] ^ im920_cast.u[5] ^ im920_cast.u[4] ^ im920_cast.u[3] ^ im920_cast.u[2] ^ im920_cast.u[1] ^ im920_cast.u[0];
    }   
}

inline void IM920::write(float val){
    if(sendDataSize <= 60){
        im920_cast.f = val;
        sprintf(im920_sendBuff, "%s%02X%02X%02X%02X",  im920_sendBuff, im920_cast.u[0],im920_cast.u[1],im920_cast.u[2],im920_cast.u[3]);
        sendDataSize += 4;
        checkSum = checkSum ^ im920_cast.u[3] ^ im920_cast.u[2] ^ im920_cast.u[1] ^ im920_cast.u[0];   
    }
}
inline void IM920::write(int val){
    if(sendDataSize <= 60){
        im920_cast.i = val;
        sprintf(im920_sendBuff, "%s%08X", im920_sendBuff, (uint32_t)val);
        sendDataSize += 4; 
        checkSum = checkSum ^ im920_cast.u[3] ^ im920_cast.u[2] ^ im920_cast.u[1] ^ im920_cast.u[0];  
    }   
}
inline void IM920::write(short val){
    if(sendDataSize <= 62){
        im920_cast.s = val;
        sprintf(im920_sendBuff, "%s%04X", im920_sendBuff, (uint16_t)val);
        sendDataSize += 2;   
        checkSum = checkSum ^ im920_cast.u[1] ^ im920_cast.u[0];
    }
}
inline void IM920::write(char val){
    if(sendDataSize < 64){
        im920_cast.c = val;
        sprintf(im920_sendBuff, "%s%02X", im920_sendBuff, (uint8_t)val); 
        sendDataSize += 1; 
        checkSum = im920_cast.u[0] ^ '0'; 
    }   
}
/*
inline void IM920::write(float *array, int count){
    if(sendDataSize < (64 - (4*count))){
        for(int i = 0; i < count; i++){
            write((float)array[i]);   
        }
    }   
}*/

inline void IM920::send(){
    if(sendDataSize > 0){
        AS = RESPONSE_STRING;
        //write(checkSum);
        //sprintf(im920_sendBuff, "%s%02X", im920_sendBuff, checkSum);
        im920_sendBuff[128] = '\0';//もしバッファがあふれて文字列になっていなかったら
        ser->printf("TXDA %s\r\n", im920_sendBuff);
        //printf("send -> TXDA %s\r\n", im920_sendBuff);
        sendDataSize = 0;//初期化
        memset(im920_sendBuff, '\0', 129);
        AS = DATA_STRING;   
    }   
}

#endif