#include "mbed.h"
#include "pqLPS22HB_lib.h"
#include "SDFileSystem.h"
#include "math.h"

const float DETECT_ALT = 100.0f;
const float PUSH_TIME = 5.0f;

/*******************************************************************************
コンストラクタ
*******************************************************************************/
Serial pc(USBTX, USBRX, 115200);
I2C i2c_bus(D4, D5);
SDFileSystem sd(D11, D12, D13, D10, "sd");
const char file_name[64] = "/sd/Hybrid_IZU201811_STMBBM_v4.csv";

pqLPS22HB_lib LPS33HW(pqLPS22HB_lib::AD0_LOW, i2c_bus);

//InterruptIn pushFlight(A0);
DigitalIn pushFlight(A0);
DigitalOut led1(D3);
DigitalOut push(A1);

Timeout timeout_stop;
Ticker tick_push;
Timer time_main;

/*******************************************************************************
関数のプロトタイプ宣言
*******************************************************************************/
void setup();

void readSensor();
void detectSeparate();

void printData();
void readPC();

void flightOn();
void pushChange();
void stopAll();

void startRec();
void stopRec();
void recData();

/*******************************************************************************
変数の宣言
*******************************************************************************/
FILE *fp;
bool save = false;
int write_c = 0;

float pres33, temp33, alt33, pres33_0, temp33_0;
bool pt33_0_set = false;

bool detect = false;
bool pushing = false;
bool flight = false;

int time_read, time_reset = 0;
int time_min, time_sec, time_ms;

/*******************************************************************************
メイン関数
*******************************************************************************/
int main() {
    setup();
    
    pc.attach(&readPC, Serial::RxIrq);
    
    pushFlight.mode(PullUp);
    //pushFlight.rise(&flightOn);
    
    time_main.reset();
    time_main.start();
    startRec();
    
    while(1) {
        readSensor();
        printData();
        recData();
        detectSeparate();
    }
}

/*******************************************************************************
セットアップ（最初に1回実行）
*******************************************************************************/
void setup(){
    wait(1.0f);
    char setup_string[512];
    
    pc.printf("\r\n**************************************************\r\n");
    pc.printf("Setting Start!!\r\n");
    strcat(setup_string, "\r\n\nSetting Start!!\r\n");
    
    ///////////////////////////////////////////LPS33HW
    LPS33HW.begin(50);
    if(LPS33HW.whoAmI() == 0){
        pc.printf("LPS33HW : OK!!\r\n");
        strcat(setup_string, "LPS33HW : OK!!\r\n");
    }
    else{
        pc.printf("LPS33HW : NG...\r\n");
        strcat(setup_string, "LPS33HW : NG...\r\n");
    }
    
    led1 = 0;
    push = 0;
    
    pc.printf("Setting Finished!!\r\n");
    strcat(setup_string, "Setting Finished!!\r\n");
    
    fp = fopen(file_name, "a");
    fprintf(fp, setup_string);
    fclose(fp);
    setup_string[0] = NULL;
    
    wait(1.0f);
}

/*******************************************************************************
センサー読み取り
*******************************************************************************/
void readSensor(){
    pres33 = LPS33HW.getPres();
    temp33 = LPS33HW.getTemp();
    alt33 = LPS33HW.getAlt(pres33_0, temp33_0);
    if(!pt33_0_set && pres33 > 900){
        pres33_0 = pres33;
        temp33_0 = temp33;
        pt33_0_set = true;
    }
    
    time_read = time_main.read_ms();
    if(time_read >= 30*60*1000){
        time_main.reset();
        time_reset ++;
    }
    time_min = time_reset * 30 + floor((double)time_read / (60 * 1000));
    time_sec = time_read % (60 * 1000);
    time_sec = floor((double)time_sec / 1000);
    time_ms = time_read % 1000;
    
    flight = pushFlight;
}

/*******************************************************************************
押し出し作動条件
    alt33 > DETECT_ALT
    pushFlight == HIGH
*******************************************************************************/
void detectSeparate(){
    if(!detect && alt33 > DETECT_ALT && flight){
        flightOn();
    }
}

/*******************************************************************************
データの表示
*******************************************************************************/
void printData(){
    pc.printf("Time : %d:%02d.%03d  ", time_min, time_sec, time_ms);
    pc.printf("LPS33HW : %.4f[hPa], %.2f[degC], %.2f[m]  ", pres33, temp33, alt33);
    pc.printf("Save : %d, Flight : %d, Detect : %d, Push : %d\r\n", save, flight, detect, pushing);
}

/*******************************************************************************
PCからの読み取り
*******************************************************************************/
void readPC(){
    char c = pc.getc();
    pc.printf("Input : %c\r\n", c);
    wait(0.25f);
    
    switch(c){
        case 'r':
        pres33_0 = pres33;
        temp33_0 = temp33;
        break;
    }
}

/*******************************************************************************
フライトピン
*******************************************************************************/
void flightOn(){
    if(!detect){
        pc.printf("Flight ON!!!!\r\n");
        tick_push.attach(&pushChange, 0.5f);
        timeout_stop.attach(&stopAll, PUSH_TIME);
        led1 = 1;
        detect = true;
    }
}

void pushChange(){
    if(push == 0){
        push = 1;
        pushing = true;
    }
    else{
        push = 0;
        pushing = false;
    }
}

void stopAll(){
    tick_push.detach();
    push = 0;
    pushing = false;
    
    stopRec();
    time_main.stop();
}

/*******************************************************************************
記録開始
*******************************************************************************/
void startRec(){
    fp = fopen(file_name, "a");
    fprintf(fp, "time,pres[hPa],temp[degC],alt[m],detect,push,flight\r\n");
    save = true;
}

/*******************************************************************************
記録終了
*******************************************************************************/
void stopRec(){
    save = false;
    fprintf(fp, "\r\n\n");
    fclose(fp);
}

/*******************************************************************************
データを記録
*******************************************************************************/
void recData(){
    if(save){
        fprintf(fp, "%d:%02d.%03d,", time_min, time_sec, time_ms);
        fprintf(fp, "%.4f,%.2f,%.2f,", pres33, temp33, alt33);
        fprintf(fp, "%d,%d,%d\r\n", detect, pushing, flight);
        
        write_c ++;
        if(write_c > 1000){
            fclose(fp);
            fp = fopen(file_name, "a");
            write_c = 0;
        }
    }
}
