/*******************************************************************************
EEPROM Library
made by T.Hayashi @PLANET-Q

1Mbit EEPROMを使うためのライブラリ
最大4素子接続でき，それをシームレスに使うためのライブラリ

使用例は一番下にあり
*******************************************************************************/
#ifndef EEPROM_LIB_H_
#define EEPROM_LIB_H_

#define EEPROM_ADDR_00_0 0xA0
#define EEPROM_ADDR_00_1 0xA8

#define EEPROM_ADDR_01_0 0xA2
#define EEPROM_ADDR_01_1 0xAA

#define EEPROM_ADDR_10_0 0xA4
#define EEPROM_ADDR_10_1 0xAC

#define EEPROM_ADDR_11_0 0xA6
#define EEPROM_ADDR_11_1 0xAE

#define EEPROM_WAIT_US 3000 /* EEPROMにデータを書き込むための待機時間(us) */


class EEPROM_lib{
    public:
    
    /***************************************************************************
    コンストラクタ
    引数
        ・i2c
        ・使用する素子数（1～4）
    例
        I2C i2c_bus(p9, p10);
        EEPROM_lib EEPROM(i2c_bus, 4);
    ***************************************************************************/
    EEPROM_lib(I2C &user_i2c, int num);
    
    /***************************************************************************
    バイト書き込み関数：writebuff
    引数
        ・書き込む素子番号（1～4）
        ・書き込むブロック（0 or 1）
        ・書き込むアドレスの上位1バイト
        ・書き込むアドレスの下位1バイト
        ・書き込むデータ（1バイト）
    例
        EEPROM.writeBuff(1, 0, 0x00, 0x00, 0xFF);
    ***************************************************************************/
    void writeByte(int num, int block, char ADDR_H, char ADDR_L, char DATA);
    
    /***************************************************************************
    ページ書き込み関数：writePage
    引数
        ・書き込み素子番号（1～4）
        ・書き込むブロック（0 or 1）
        ・書き込むアドレスの上位1バイト
        ・書き込むアドレスの下位1バイト
        ・書き込むデータのポインタ（データの配列）（配列はchar型でサイズは128）
    例
        char DATA[128];
        EEPROM.writePage(1, 0, 0x00, 0x00, DATA);
    ***************************************************************************/
    void writePage(int num, int block, char ADDR_H, char ADDR_L, char *DATA);
    
    
    /***************************************************************************
    連続書き込みの最初のアドレス指定関数：setWriteAddr
    引数
        ・開始素子番号（1～4）
        ・開始ブロック（0 or 1）
        ・開始アドレスの上位1バイト
        ・開始アドレスの下位1バイト
    例
        EEPROM.setWriteAddr(1, 0, 0x00, 0x00);
    ***************************************************************************/
    void setWriteAddr(int num, int block, char ADDR_H, char ADDR_L);
    
    
    /***************************************************************************
    後でまとめて書き込むためのバッファに値を溜める関数：chargeBuff
    返り値
        バッファ内で何バイト目まで埋まっているか（1～128）
    引数
        各型のデータ
    例
        int ptr;
        ptr = EEPROM.chargeBuff((char)0x32);
        ptr = EEPROM.chargeBuff((bool)true);
        ptr = EEPROM.chargeBuff((int)12);
        ptr = EEPROM.chargeBuff((short)23);
        ptr = EEPROM.chargeBuff((long long)99999999999);
        ptr = EEPROM.chargeBuff((float)12.3);
        ptr = EEPROM.chargeBuff((double)12.3456);
        ※変数も可
        ※型名を記述すること
    ***************************************************************************/
    int chargeBuff(char DATA);
    int chargeBuff(bool DATA);
    int chargeBuff(int DATA);
    int chargeBuff(short DATA);
    int chargeBuff(long long DATA);
    int chargeBuff(float DATA);
    int chargeBuff(double DATA);
    
    /***************************************************************************
    バッファの内容を実際に書き込む関数：writeBuff
    例
        EEPROM.writeBuff();
    ***************************************************************************/
    void writeBuff();
    
    /***************************************************************************
    書き込むアドレスを次のページに移動する関数：setNextPage
    返り値
        次のぺージの開始アドレス
        ※（上位から）1バイト目：素子番号（0x00 0x01 0x02 0x03）
        ※2バイト目：ブロック（0x00 0x01）
        ※3,4バイト目：アドレス
        ※例えば，0x01001280だった場合，
            素子番号2，ブロック0，アドレス0x1280が次に書き込むアドレス
    例
        int ptr = EEPROM.setNextPage();
    ***************************************************************************/
    int setNextPage();
    
    /***************************************************************************
    バイト読み取り関数：readByte
    返り値
        読み取ったデータ
    引数
        ・読み取る素子番号（1～4）
        ・読み取るブロック（0 or 1）
        ・読み取るアドレスの上位1バイト
        ・読み取るアドレスの下位1バイト
    例
        char data = EEPROM.readByte(1, 0, 0x00, 0x00);
    ***************************************************************************/
    char readByte(int num, int block, char ADDR_H, char ADDR_L);
    
    /***************************************************************************
    複数バイト読み取り関数：readMultiByte
    引数
        ・読み取る素子番号（1～4）
        ・読み取るブロック（0 or 1）
        ・読み取るアドレスの上位1バイト
        ・読み取るアドレスの下位1バイト
        ・読み取ったデータを格納する配列のポインタ
        ・読み取るデータのサイズ
    例
        char DATA[128];
        EEPROM.readMultiByte(1, 0, 0x00, 0x00, DATA, 128);
    ***************************************************************************/
    void readMultiByte(int num, int block, char ADDR_H, char ADDR_L, char *DATA, int size);
    
    
    private:
    I2C *i2c;
    char slave;
    
    union EEPROM_CAST{
        long long ll;
        int i;
        short s;
        float f;
        double d;
        char c;
        bool b;
        char cast[8];
    };
    union EEPROM_CAST EEPROM_cast;
    
    char cmd[130]; //I2C通信用コマンド配列
    int max_num;
    
    int buff_ptr; //バッファの何個目まで入っているか
    
    char ptr[4];
};

#endif

/*******************************************************************************
example

#include "EEPROM_lib.h"

Serial PC(USBTX, USBRX, 115200);
I2C i2c_bus(p9, p10);

EEPROM_lib EEPROM(i2c_bus, 4);

int main(){
    //////////////////////////////////////////write
    EEPROM.setWriteAddr(1, 0, 0x00, 0x00);
    int ptr, n = 0;
    while(1){
        while(1){
            ptr = EEPROM.writeBuff((int)n++);
            if(ptr == 128){
                EEPROM.writeBuff();
                ptr = EEPROM.setNextPage();
                break;
            }
        }
        if(ptr == 0x04000000){
            break;
        }
    }
    
    ///////////////////////////////////////read
    char data[128];
    int num, block;
    char ADDR_H, ADDR_L;
    
    num = 1;
    block = 0;
    ADDR_H = ADDR_L = 0x00;
    
    for(int i = 0; i < 4096; i ++){
        switch(i){
            case 512:
            num = 1;
            block = 1;
            break;
            
            case 1024:
            num = 2;
            block = 0;
            break;
            
            case 1536:
            num = 2;
            block = 1;
            break;
            
            case 2048:
            num = 3;
            block = 0;
            break;
            
            case 2560:
            num = 3;
            block = 1;
            break;
            
            case 3072:
            num = 4;
            block = 0;
            break;
            
            case 3584:
            num = 4;
            block = 1;
            break;
        }
        
        EEPROM.readMultiByte(num, block, ADDR_H, ADDR_L, data, 128);
        for(int x = 0; x < 128; x ++){
            pc.printf("%02x ", data[x]);
        }
        pc.printf("\r\n");
        
        if(ADDR_L == 0x00){
            ADDR_L = 0x80;
        }
        else{
            ADDR_L = 0x00;
            ADDR_H ++;
        }
    }
    
}

*******************************************************************************/