#include "mbed.h"
#include "SDFileSystem.h"

Timer timer;

DigitalIn button(PC_13, PullUp);

SDFileSystem sd(PA_7, PA_6, PA_5, PB_6, "sd", PA_8, SDFileSystem::SWITCH_NONE, 25000000);
char buffer[4096];

void writeTest()
{
    //Test write performance by creating a 1MB file
    printf("Testing %iB write performance...", sizeof(buffer));
    FileHandle* file = sd.open("Test File.bin", O_WRONLY | O_CREAT | O_TRUNC);
    if (file != NULL) {
        timer.start();
        for (int i = 0; i < (1048576 / sizeof(buffer)); i++) {
            if (file->write(buffer, sizeof(buffer)) != sizeof(buffer)) {
                timer.stop();
                printf("write error!\n");
                timer.reset();
                return;
            }
        }
        timer.stop();
        if (file->close())
            printf("failed to close file!\n");
        else
            printf("done!\n\tResult: %.2fKB/s\n", 1024 / (timer.read_us() / 1000000.0));
        timer.reset();
    } else {
        printf("failed to create file!\n");
    }
}

void readTest()
{
    //Test read performance by reading the 1MB file created by writeTest()
    printf("Testing %iB read performance...", sizeof(buffer));
    FileHandle* file = sd.open("Test File.bin", O_RDONLY);
    if (file != NULL) {
        timer.start();
        int iterations = 0;
        while (file->read(buffer, sizeof(buffer)) == sizeof(buffer))
            iterations++;
        timer.stop();
        if (iterations != (1048576 / sizeof(buffer)))
            printf("read error!\n");
        else if (file->close())
            printf("failed to close file!\n");
        else if (sd.remove("Test File.bin"))
            printf("failed to delete file!\n");
        else
            printf("done!\n\tResult: %.2fKB/s\n", 1024 / (timer.read_us() / 1000000.0));
        timer.reset();
    } else {
        printf("failed to open file!\n");
    }
}

int main()
{
    //Configure CRC, large frames, and write validation
    sd.crc(true);
    sd.large_frames(true);
    sd.write_validation(true);

    //Fill the buffer with random data for the write test
    srand(time(NULL));
    for (int i = 0; i < sizeof(buffer); i++)
        buffer[i] = rand();

    while(1) {
        //Simple button debouncing
        wait(0.5);

        //Print the start message
        printf("\nPress the button to perform tests: ");

        //Wait for the button to be pressed
        while(button);

        //Make sure a card is present
        if (!sd.card_present()) {
            printf("\nNo card present!\n");
            continue;
        }

        //Try to mount the SD card
        printf("\nMounting SD card...");
        if (sd.mount() != 0) {
            printf("failed!\n");
            continue;
        }
        printf("success!\n");

        //Display the card type
        printf("\tCard type: ");
        SDFileSystem::CardType cardType = sd.card_type();
        if (cardType == SDFileSystem::CARD_NONE)
            printf("None\n");
        else if (cardType == SDFileSystem::CARD_MMC)
            printf("MMC\n");
        else if (cardType == SDFileSystem::CARD_SD)
            printf("SD\n");
        else if (cardType == SDFileSystem::CARD_SDHC)
            printf("SDHC\n");
        else
            printf("Unknown\n");

        //Display the card capacity
        printf("\tSectors: %u\n", sd.disk_sectors());
        printf("\tCapacity: %.1fMB\n", sd.disk_sectors() / 2048.0);

        /*//Format the card
        printf("Formatting SD card...");
        if (sd.format() != 0) {
            printf("failed!\n");
            continue;
        }
        printf("success!\n");*/

        //Perform a read/write test
        writeTest();
        readTest();

        //Unmount the SD card
        sd.unmount();

        printf("Hello World!\n");

        mkdir("/sd/mydir", 0777);

        FILE *fp = fopen("/sd/mydir/sdtest.txt", "w");
        if(fp == NULL) {
            error("Could not open file for write\n");
        }
        fprintf(fp, "Hello fun SD Card World!");
        fclose(fp);

        printf("Goodbye World!\n");

    }
}
