/* Armour Program 

Authors: Jamieson Brynes and Qi Hu

This program is designed for use with the gun and scoreboard program from the Laser Tag - University of Oxford CWM 2015 project.

A few notes: 

For sending data to the gun
'a' ==> Enable gun
'z' ==> Send back bullet count data
'B' ==> Disable gun

For data received from the IR
'a' ==> Player 1
'z' ==> Player 2
'B' ==> Player 3
52 ==> Static target ( does not contribute to score)

*/

#include "mbed.h"
#include "comm.h"

DigitalOut myled(LED1);
Serial RX(NC,PTE1);              //Set up serial for IR sensor
DigitalOut LEDHigh(PTE30);       //Pin for LED high value
DigitalOut LEDLow(PTE29);        //Pin for LED low value
Serial gun(PTE22,PTE23);         //Serial communication between the armour and the gun
Serial pc(USBTX, USBRX);         //Pc serial for debugging
DigitalOut gunEnableFlag(PTE21); //Flag for gunEnable

//Initialize functions
void checkHit();
void checkScoreboard();
void gameStart();
void gameEnd();
void gamePause();
void gameContinue();
void powerDown();
void sendGunInfo(char code);
void powerUp();

//Initialize communication buffers
char* rxBuffer = new char[4];
char* txBuffer = new char[4];


char gameOn;            //Initialize boolean for game state

Timer t;               //Timer for the power down situation


int main()
{
    
    //Initialize txBuffer
    for (int i = 0; i < 4; i++) {
        txBuffer[i] = 0;
    }

    RX.baud(1000);          //Set baud rate for the receiver

    //Set default values for the LEDs and game state
    LEDLow = 0;
    LEDHigh = 0;
    gameOn = 0;

    
    commInit(3);             //Initialize communications

while(true)
{


    while(gameOn) {
       
        //Game loop
        checkHit();
        checkScoreboard();
    }

    while(!gameOn) {
        
        //Loop to run when game is off
        checkScoreboard();
        if(t.read() >  5)       //Check to see if player has been dead for long enough
        {
            powerUp();
            t.stop();
            t.reset();
        }
    }
    
    }

}

void checkHit()
{
    char identifier = 0;


    //Logic to see if we have been hit
    if(RX.readable()) {
        //Read until we find a value we recognize
        while(identifier != 'a' && identifier != 'z' && identifier != 'B' && identifier != 52) {
            identifier = RX.getc();
        }
    }

    if(identifier != 0) {
        //Store data

        if(identifier == 'a') {
            txBuffer[0]++;
        }
        if(identifier == 'z') {
            txBuffer[1]++;
        }
        if(identifier == 'B') {
            txBuffer[2]++;
        }


        powerDown();


    }
    

    //If not hit, continue as normal


}

void checkScoreboard()
{
// comm check, needs to be done regularly
    if(commRead(rxBuffer)) {

        //Get gun shots and load it in
        sendGunInfo('z');
        wait(0.5);
        if(gun.readable())
        {
            char shots;
            shots = gun.getc();
            txBuffer[3] += shots;
            
            while(gun.readable())
            {
                char trash;
                trash = gun.getc();
            }
        }
        
        commSetAck(txBuffer);
        pc.printf("Refreshing acknowledge buffer \n\r");
        switch(rxBuffer[0]) 
        {
            case 0: //code to be executed on start signal
                pc.printf("Starting\n\r");
                gameStart();
                //wait(2);

                break;
            case 1: // code to be executed on a get hits request
                 pc.printf("Received get request.\n\r");   
                txBuffer[3] = 0;
                
                break;
            case 2: // code to be executed on end signal 
                pc.printf("Ending game");
                gameEnd();

                break;
            case 3: // code to be executed on pause signal 

                gamePause();

                break;
            case 4: // code to be executed on resume signal 

                gameContinue();

                break;
            default:
                commSetAck(txBuffer);
                
                break;

        }
    }

}

void powerDown()
{

    sendGunInfo('B');
    LEDHigh = 0;
    gameOn = 0;
    
    //Time delay for the armour to be down
    t.start();

}

void powerUp()
{
    //Switch LEDs on
    LEDHigh = 1;
    //Enable gun
    sendGunInfo('a');
    
    gameOn = 1;
 
    

    while(RX.readable()) 
    {
        //Loop to wipe buffer of the receiver
        //Need to ignore any shots received during the power down phase
        char trash;
        trash = RX.getc();
    }
}

void gameStart()
{
    pc.printf("Starting game\n\r");
    //Enable gun
    sendGunInfo('a');

    //Switch LEDs on
    LEDHigh = 1;
    
    //Enable game loop
    gameOn = 1;
    
    //Reset scoring buffer
    for (int i = 0; i < 4; i++) {
        txBuffer[i] = 0;
    }
    
    pc.printf("Ending start game routine\n\r");
    
    

}


void gameEnd()
{
    //Set flag and send data to gun
    gameOn = 0;
    sendGunInfo('B');

    //Turn off LED
    LEDHigh = 0;
    
    while(RX.readable()) 
    {
        //Loop to wipe buffer of the receiver
        //Need to ignore any shots received during the power down phase
        char trash;
        trash = RX.getc();
    }
}

void gamePause()
{
    //Set flag and send data to gun
    gameOn = 0;
    sendGunInfo('B');
    t.stop();


}

void gameContinue()
{
    //Set flag and send data to gun
    gameOn = 1;
    sendGunInfo('a');
    
    while(RX.readable()) 
    {
        //Loop to wipe buffer of the receiver
        //Need to ignore any shots received during the power down phase
        char trash;
        trash = RX.getc();
    }
    
    if(t.read() > 0)
    {
        t.start();
    }

}

void sendGunInfo(char code)
{
    //Send flag to gun and push a code to them
    gunEnableFlag = 1;
    gun.putc(code);
    gunEnableFlag = 0;
}