/*
** This software can be freely used, even comercially, as highlighted in the license.
** 
** Copyright 2014 GHI Electronics, LLC
** 
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
** 
**     http://www.apache.org/licenses/LICENSE-2.0
** 
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
**
**/

#include "mbed.h"
#include "TextLCD.h"

SPI spi(P0_21, P0_22, P1_15);                                   // mosi, miso, sclk
TextLCD lcd(P0_23, P0_16, P0_14, P0_13, P0_12, P0_11);          // RS, E, D4-D7, LCDType=LCD16x2, BL=NC, E2=NC, LCDTCtrl=HD44780 

DigitalOut compassLED[] = {(P0_19), (P0_9), (P0_18), (P0_8)};   // The South, West, North, East LEDs
DigitalOut OutputEnable(P0_5);
DigitalOut Latch(P0_4);
DigitalOut Buzzer(P0_22);

DigitalIn userButton1(P0_17, PullUp);
InterruptIn  selectButton(P0_1);

enum CompassPoints { SOUTH = 0, WEST = 1, NORTH = 2, EAST = 3 };

void initilize();
void printToLCD(const char* text);
void RotateLEDs(float delay);
void SetLED(uint32_t LEDBitMap);
bool ReadPlayerButton();
bool ReadSelectButton();
void soundBuzzer(int Hz);
void splash();
void gameLoop();
int checkTargetHit(int currentLED);
void displayHighestLevel();
void score();
void winner();

int level;
int highestLevel;
int rotateFactor;
int LEDBitmap;
char debugString [50];

void initilize()
{
    level = 1;
    highestLevel = 1;
    rotateFactor = 5;
    LEDBitmap = 1;

    OutputEnable = false;
    selectButton.mode(PullUp);
    selectButton.rise(&displayHighestLevel);
    
    lcd.putc('R');
    lcd.putc('o');
    lcd.putc('u');
    lcd.putc('l');
    lcd.putc('e');
    lcd.putc('t');
    lcd.putc('t');
    lcd.putc('e');
    
    spi.format(16,0);                   // Sets the bits to 16 to write; sets the clock's Polarity and Phase
    spi.frequency(10 * 1000 * 1000);    // Sets SPI clock to 20,000,000 Hz or 10MHz
}

void printToLCD(const char* text)
{
    // Will add a text scrolling function here if the text is
    // greater than 8 characters.
    if (strlen(text) > 8) {
        return;
    }
    
    int x = 0;
    lcd.cls();

    while (text[x] != '\0') {
        lcd.putc(text[x]);
        x++;
    }
}

void SetLED(uint32_t LEDBitMap)
{
    uint16_t shiftMap = 0;

    // Ring Segments - all on linked shift registers
    shiftMap = (LEDBitMap >> 1) &0xf;
    shiftMap |= ((LEDBitMap >> 6) &0xf) << 4;
    shiftMap |= ((LEDBitMap >> 11) &0xf) << 8;
    shiftMap |= ((LEDBitMap >> 16) &0xf) << 12;
    
    spi.write(shiftMap);

    Latch = true;
    Latch = false;

    // Set the LEDs connected directly to the processor   
    compassLED[SOUTH] = LEDBitMap &0x1;
    compassLED[WEST] = (LEDBitMap >> 5) &0x1;
    compassLED[NORTH] = (LEDBitMap >> 10) &0x1;
    compassLED[EAST] = (LEDBitMap >> 15) &0x1;
}

bool ReadPlayerButton()
{
    // Logic High = True, Logic Low = False.
    // userButton1 is set with an internal Pull-up which makes
    // the button true by default and false when pressed.
    // Must invert to make the read button function true.
    return !userButton1; // Logic inverted
}

bool ReadSelectButton()
{
    // Logic High = True, Logic Low = False.
    // selectButton is set with an internal Pull-up which makes
    // the button true by default and false when pressed.
    // Must invert to make the read button function true.
    return !selectButton; // Logic inverted
}

void soundBuzzer(int Hz)
{
    int ticks = Hz/64;
    int tickCount = 0;
    float frequency = 1/(float)Hz;
    
    while(tickCount < ticks) {
        wait(frequency);
        Buzzer = true;
        wait(frequency);
        Buzzer = false;
        tickCount++;
    }
}

void RotateLEDs(float delay)
{
        if (LEDBitmap == (1 << 10)) {
            soundBuzzer(300);
        }
        
        SetLED(LEDBitmap);
        
        LEDBitmap <<= 1;
        wait(delay);
        if(LEDBitmap >= (1 << 20)) {
            LEDBitmap = 1;
        }
 }

void splash()
{
    int LEDBitmap = 1;
    
    for (int x = 1; x <= 20; x++) {
        LEDBitmap <<= 1;
        SetLED(LEDBitmap);
        soundBuzzer((x * 100) + 500);
        wait(0.0078125);
    }
    
    for (int y = 0; y < 5; y++) {
        SetLED(0xfffff);
        soundBuzzer(600);
        wait(.03125);
        SetLED(0x0);
        wait(.03125);
    }
}

int checkTargetHit(int currentLED)
{
    wait(.5);
    if (currentLED == (1 << 11)) {
        return 3;
    } else if (currentLED == (1 << 10) || currentLED == (1 << 12)) {
        return 2;
    } else if (currentLED == (1 << 9) || currentLED == (1 << 13)) {
        return 1;
    } else {
        return 0;
    }
}

void displayHighestLevel()
{
    sprintf (debugString, "H Lev %d",highestLevel);
    printToLCD(debugString);
    wait(2.5);
    sprintf (debugString, "Level %d",level);
    printToLCD(debugString);
}

void score()
{
    int target = checkTargetHit(LEDBitmap);
    
    switch (target) {
        case 3:
            level += 3;
            break;
        case 2:
            if(level < 10) {
                level += 2;
            } else {
                level -= 2;
            }
            break;
        case 1:
            level += 1;
            break;
        default:
            level--;
            if (level <= 1) {
                level = 1;
            }
    }
    
    if (level > highestLevel) {
        highestLevel = level;
    }
}

void winner()
{
    int winCounter = 1;
    while (winCounter++ <= 5) {
        printToLCD("Winner!!");
        SetLED(0xFFFFF);
        wait(.5);
        printToLCD("");
        SetLED(0x0);

        for (int x = 1; x <= 10; x++) {
            soundBuzzer((x * 100) + 500);
            wait(0.0078125);
        }
        
    }
    
    while (true) {
        printToLCD("Winner!!");
        SetLED(0xFFFFF);
        wait(.5);
    }

}

void gameLoop()
{
    while (true) {
        sprintf (debugString, "Level %d", level);
        printToLCD(debugString);
        
        while (!ReadPlayerButton()) {
            RotateLEDs(1/(((float)level * rotateFactor) + 19));
        }
        
        score();
        
        if (level > 99) {
            winner();
        }
        
        while (ReadPlayerButton()) { // Will hold game if hold on to button to prevent cheating
        }
    }
}

int main()
{
    initilize();
    splash();
    gameLoop();
}
