#ifndef VT100_H
#define VT100_H included
#include "mbed.h"

/** vt100 class
 * Utility for handling text/letter on a terminal
 * which can handle VT100 escape command sequence.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "vt100.h"
 * 
 * vt100 tty ;
 * 
 * int main() {
 *   int count = 0 ;
 *   tty.cls() ;
 *   tty.black() ;
 *   tty.frame(5, 5, 15, 9) ;
 *   while(1) {
 *       tty.locate(7, 7) ;
 *       tty.setFG(count % 8) ;
 *       printf("%d\r\n", count++) ;
 *       wait(1.0) ;
 *   }
 * }
 * @endcode
 * 
 * Note: I know there should be other libraries
 *  with similar functions, but I could not help 
 *  writing one for myself anyway.
 */

class vt100 : public Serial {
public:
    /** constructor 
     * @param baud baud rate
     */
    vt100(int baud = 115200) ;
    
    vt100(PinName tx_pin, PinName rx_pin, int baud=115200) ;
    
    /** destructor */
    ~vt100() ;
    
    /** clear screen */
    void cls(void) ;
    
    /** move cursor to (x, y) 
     * @param x start column of the next letter
     * @param y start row of the next letter
     * @note no value checking is performed.
     */
    void locate(int x, int y) ;
    
    /** print a letter c at (x,y)
     * @param c the letter to be written 
     * @param x column of the letter
     * @param y row of the letter
     */
    void putChar(int x, int y, char c) ;
    
    /** print a string str from (x,y) 
     * @param *str c-style string to be written
     * @param x column of the first letter
     * @param y row of the first letter
     */
    void putStr(int x, int y, char *str) ;
    
    /** print a line of char 
     * @param c the letter to form the line
     * @param x1 starting column
     * @param y1 starting row
     * @param x2 ending column
     * @param y2 ending row
     */
    void line(int x1, int y1, int x2, int y2, char c='*') ;
    
    /** print a text frame 
     * @param x1 left column
     * @param y1 top row
     * @param x2 right column
     * @param y2 bottom row
     */
    void frame(int x1, int y1, int x2, int y2) ;
    
    /** print a text circle
     * @c the letter to form the circle
     * @param x0 center column
     * @param y1 center row
     * @param r radius
     */
    void circle(int x0, int y0, int r, char c='*') ;
    
    /** set foreground color
     * @param newFG new foreground color
     * @returns previous foreground color
     * @note 0 BLACK
     * @note 1 RED
     * @note 2 GREEN
     * @note 3 YELLOW
     * @note 4 BLUE
     * @note 5 PURPLE
     * @note 6 CIAN
     * @note 7 WHITE
     */
    int setFG(int newFG) ;
    
    /** get current foreground color 
     * @returns current foreground color
     */
    int getFG(void) ;
    
    /** set background color
     * @param newBG new background color
     * @returns previous background color
     */
    int setBG(int newBG) ;
    
    /** get current background color 
     * @returns current background color
     */
    int getBG(void) ;
    /** set foreground color to black */
    void black(void) ;
    /** set foreground color to red */
    void red(void) ;
    /** set foreground color to green */
    void green(void) ;
    /** set foreground color to yellow */
    void yellow(void) ;
    /** set foreground color to blue */
    void blue(void) ;
    /** set foreground color to purple */
    void purple(void) ;
    /** set foreground color to cian */
    void cian(void) ;
    /** set foreground color to white */
    void white(void) ;
protected:
private:
    int foreground ;
    int background ;

} ;

#endif /* VT100_H */