#include "mbed.h"
#include "PSE530.h"

/**
 * SMC PSE530 pressure sensor
 * analog output 1.0V - 5.0V
 * 1.0V : 0
 * 5.0V : 1MPa
 * (at 0.6V : -0.1MPa)
 * Our sensor I/F converts 0-5V to 0-3V
 * So we suppose V = Analog Float Value : Pressure
 * 0.6V = 0.2 : 0
 * 3.0V = 1.0 : 1MPa
 */
 
 /**
  * conversion from Pa to kgf/cm2
  * 98,066.5 Pa = 1 kgf/cm2
  * 1 Pa = 1 / 98066.6 kgf/cm2
  */

PSE530::PSE530(AnalogIn *ain)
{
    _ain = ain ;
}

PSE530::~PSE530(void) 
{
    if (_ain) {
        delete _ain ;
    }
}

/**
 * On FRDM-KL25Z ADC's AREF is about 3.28V
 * Where the converted pressure output is 0 to 3.21V
 * So we must map ADC output 0 to 3.21/3.28 as full scale
 * 
 * Then according to the datasheet of PSE530
 * when full range is 0V to 5V
 * 1V is 0 and 5V is 1MPa which is converted to
 * 0.642/3.28 to 3.21/3.28 ~ 0.195731 to 0.9786585.
 * The linear equation of
 * y = a x + b
 * 0 = a * 0.195731 + b
 * 1 = a * 0.978658 + b
 * results a = 1.277, b = -0.250
 */
float PSE530::getPressure(void)
{
    float coef_A = 1.277 ;
    float coef_B = -0.250 ;
    float av = 0.0 ;
    float value = 0.0 ;
    av = coef_A * _ain->read() + coef_B ;
//    printf("Pressure ADC = %.4f\n", av) ;
    value = 1000000 * av  ; /* 1MPa at 1.0 */
    value = value / 98066.5 ; /* Pa -> kgf/cm2 */
    return( value ) ;
}
