/**
 * MMA8451Q 3-Axis, 14-bit/8-bit Digital Accelerometer
 */
 
#include "mbed.h"
#include "MMA8451Q.h"

#define REG_STATUS           0x00 // when F_MODE = 00
#define REG_FIFO_STATUS      0x00 // when F_MODE > 0
#define REG_XYZ_FIFO         0x01 // Root pointer to XYZ FIFO data
#define REG_OUT_X_MSB        0x01 // 8 MSBs of 14-bit sample
#define REG_OUT_X_LSB        0x02 // 6 LSBs of 14-bit sample
#define REG_OUT_Y_MSB        0x03 
#define REG_OUT_Y_LSB        0x04
#define REG_OUT_Z_MSB        0x05
#define REG_OUT_Z_LSB        0x06
#define REG_F_SETUP          0x09 // FIFO setup
#define REG_TRIG_CFG         0x0A // Map of FIFO daa capture events
#define REG_SYSMOD           0x0B // Current System Mode
#define REG_INT_SOURCE       0x0C // Interrupt status
#define REG_WHO_AM_I         0x0D // Device ID (0x1A)
#define REG_XYZ_DATA_CFG     0x0E // Dynamic Range Settings
#define REG_HP_FILTER_CUTOFF 0x0F // Cutoff freq is set to 16Hz@800Hz
#define REG_PL_STATUS        0x10 // Landscape/Portrait orientation status
#define REG_PL_CFG           0x11 // Landscape/Portrait configuration
#define REG_PL_COUNT         0x12 // Landscape/Portrait debounce counter 
#define REG_PL_BF_ZCOMP      0x13 // Back/Front, Z-Lock Trip threshold
#define REG_P_L_THS_REG      0x14 // Portrait to Landscape Trip Angle is 29 degree
#define REG_FF_MT_CFG        0x15 // Freefall/Motion function block configuration
#define REG_FF_MT_SRC        0x16 // Freefall/Motion event source register
#define REG_FF_MT_THS        0x17 // Freefall/Motion threshold register
#define REG_FF_MT_COUNT      0x18 // Freefall/Motion debounce counter
// TRANSIENT
#define REG_TRANSIENT_CFG    0x1D // Transient functional block configuration
#define REG_TRANSIENT_SRC    0x1E // Transient event status register
#define REG_TRANSIENT_THS    0x1F // Transient event threshold
#define REG_TRANSIENT_COUNT  0x20 // Transient debounce counter
// PULSE
#define REG_PULSE_CFG        0x21 // ELE, Double_XYZ or Single_XYZ
#define REG_PULSE_SRC        0x22 // EA, Double_XYZ or Single_XYZ
#define REG_PULSE_THSX       0x23 // X pulse threshold
#define REG_PULSE_THSY       0x24 // Y pulse threshold
#define REG_PULSE_THSZ       0x25 // Z pulse threshold
#define REG_PULSE_TMLT       0x26 // Time limit for pulse
#define REG_PULSE_LTCY       0x27 // Latency time for 2nd pulse
#define REG_PULSE_WIND       0x28 // Window time for 2nd pulse
#define REG_ASLP_COUNT       0x29 // Counter setting for Auto-SLEEP
// Control Registers
#define REG_CTRL_REG1        0x2A // ODR = 800Hz, STANDBY Mode
#define REG_CTRL_REG2        0x2B // Sleep Enable, OS Modes, RST, ST
#define REG_CTRL_REG3        0x2C // Wake from Sleep, IPOL, PP_OD
#define REG_CTRL_REG4        0x2D // Interrupt enable register
#define REG_CTRL_REG5        0x2E // Interrupt pin (INT1/INT2) map
// User Offset
#define REG_OFF_X            0x2F // X-axis offset adjust
#define REG_OFF_Y            0x30 // Y-axis offset adjust
#define REG_OFF_Z            0x31 // Z-axis offset adjust

// Value definitions
#define BIT_TRIG_TRANS       0x20  // Transient interrupt trigger bit
#define BIT_TRIG_LNDPRT      0x10  // Landscape/Portrati Orientation
#define BIT_TRIG_PULSE       0x08  // Pulse interrupt trigger bit
#define BIT_TRIG_FF_MT       0x04  // Freefall/Motion trigger bit

MMA8451Q::MMA8451Q(I2C *i2c, int addr) : m_addr(addr<<1) {
    // activate the peripheral
    p_i2c = i2c ;
    uint8_t data[2] = {REG_CTRL_REG1, 0x01};
    writeRegs(data, 2);
}

MMA8451Q::~MMA8451Q() { }

int MMA8451Q::readRegs(int addr, uint8_t * data, int len) 
{
    char t[1] = {addr};
    int result ;
    result = p_i2c->write(m_addr, t, 1, true);
    if (result == 0) {
        result = p_i2c->read(m_addr, (char *)data, len);
    }
    return( result ) ;
}

int MMA8451Q::writeRegs(uint8_t * data, int len) 
{
    int result ;
    result = p_i2c->write(m_addr, (char *)data, len);
    return( result ) ;
}

int MMA8451Q::getAllRawData(int16_t value[])
{
    int result ;
    uint8_t data[6] ;
    result = readRegs(REG_OUT_X_MSB, data, 6) ;
    if (result == 0) {
        value[0] = ((int16_t)((data[0] << 8) | data[1])) >> 2 ;
        value[1] = ((int16_t)((data[2] << 8) | data[3])) >> 2 ;
        value[2] = ((int16_t)((data[4] << 8) | data[5])) >> 2 ;
    }
    return( result ) ;
}

int MMA8451Q::getAllData(float fvalue[])
{
    int result ;
    uint8_t data[6] ;
    result = readRegs(REG_OUT_X_MSB, data, 6) ;
    if (result == 0) {
        fvalue[0] = (float)((int16_t)((data[0] << 8) | data[1])) / 16384.0 ;
        fvalue[1] = (float)((int16_t)((data[2] << 8) | data[3])) / 16384.0 ;
        fvalue[2] = (float)((int16_t)((data[4] << 8) | data[5])) / 16384.0 ;
    }
    return( result ) ;
}

int16_t MMA8451Q::getRawData(uint8_t addr)
{
    int16_t value ;
    uint8_t data[2] ;
    readRegs(addr, data, 2) ;
    value = ((int16_t)((data[0] << 8) | data[1])) >> 2 ;
    return( value ) ;
}

int16_t MMA8451Q::getRawX(void) 
{
    int16_t value ;
    value = getRawData(REG_OUT_X_MSB) ;
    return( value ) ;
}

int16_t MMA8451Q::getRawY(void) 
{
    int16_t value ;
    value = getRawData(REG_OUT_Y_MSB) ;
    return( value ) ;
}

int16_t MMA8451Q::getRawZ(void) 
{
    int16_t value ;
    value = getRawData(REG_OUT_Z_MSB) ;
    return( value ) ;
}
  
float MMA8451Q::getAccX(void)
{
    return(((float)getRawX())/4096.0) ;
}

float MMA8451Q::getAccY(void) 
{
    return(((float)getRawY())/4096.0) ;
}

float MMA8451Q::getAccZ(void) 
{
    return(((float)getRawZ())/4096.0) ;
}