#include "mbed.h"
#include "edge_mgr.h"
#include "af_attributes.h"

#include "edge_time.h"
#include "edge_pin.h"
#include "MMA8451Q.h"
#include "VEML6040.h"
#include "LM75B.h"
#include "SMTC502AT.h"
#include "PSE530.h"
#include "SO1602A.h"

#include "edge_sensor.h"
#include "edge_accel.h"
#include "edge_color.h"
#include "edge_temp.h"
#include "edge_pressure.h"

#define MMA8451Q_I2C_ADDRESS 0x1C
#define VEML6040_I2C_ADDRESS 0x10
#define LM75B_I2C_ADDRESS    0x48
#define SO1602A_I2C_ADDRESS  0x3C

#define NUM_MAX_SENSOR 5

bool            verbos = true ;
edge_sensor     *sensor[NUM_MAX_SENSOR] ;
int             num_sensor   = 0 ;

edge_accel      *accel       = 0 ;
edge_color      *color[2]    = {0, 0} ;
edge_temp       *temp        = 0 ;
edge_pressure   *pressure    = 0 ;

PwmOut          *led[3]      = {0, 0, 0} ;
uint16_t        pwm[3]       = { 0x5FA2, 0xB09B, 0x83DF } ;
I2C             *edge_i2c0   = 0 ;
I2C             *edge_i2c1   = 0 ;
SO1602A         *display     = 0 ; /* OLED display on I2C */
MMA8451Q        *mma8451q    = 0 ;
VEML6040        *veml6040[2] = { 0, 0 } ;
LM75B           *lm75b0      = 0 ; /* for temp1 */
AnalogIn        *an0         = 0 ; /* for temp2 */
SMTC502AT       *smtc502at0  = 0 ;
AnalogIn        *an1         = 0 ; /* for temp3 */
SMTC502AT       *smtc502at1  = 0 ;
LM75B           *lm75b1      = 0 ; /* for temp4 */
AnalogIn        *an2         = 0 ; /* for gas pressure */
PSE530          *pse530      = 0 ; /* gas pressure sensor */

static int loop_interval     = 100 ; // 1000 ; 
static int accel_interval    = 10 ;
int        first_loop        = 1 ;

int init_edge_attribute(void)
{
    static int sensor_index = 0 ;
    if (first_loop) {
        enable_sensors() ;
        first_loop = 0 ;
        printf("Sensor loop started!\n") ;
        if (display) {
            display->cls() ;
            display->locate(0, 0) ;
            display->putStr("Sensor Loop") ;
            display->locate(2, 1) ;
            display->putStr("Started!") ;
        }
    }
    switch(sensor_index) {
    case 0: /* accel */
        if (reset_reason_str) {
            afero->setAttribute(ATTR_MCU_RESET_REASON, reset_reason_str) ;
            reset_reason_str = 0 ;
        }
        if (sensor[sensor_index]) {
           afero->setAttributeBool(ATTR_ACCEL_PRESENT, true) ;
           afero->setAttributeBool(ATTR_ACCEL_ENABLE, true) ;
        } else {
            afero->setAttributeBool(ATTR_ACCEL_PRESENT, false) ;
        }
        break ;
    case 1: /* color0 */
        if (sensor[sensor_index]) {
            afero->setAttributeBool(ATTR_COLOR0_PRESENT, true) ;
            afero->setAttributeBool(ATTR_COLOR0_ENABLE, true) ;
            afero->getAttribute(ATTR_COLOR0_PWM_R) ;
            afero->getAttribute(ATTR_COLOR0_PWM_G) ;
            afero->getAttribute(ATTR_COLOR0_PWM_B) ;
        } else {
            afero->setAttributeBool(ATTR_COLOR0_PRESENT, false) ;
        }
        break ;
    case 2: /* color1 */
        if (sensor[sensor_index]) {
            afero->setAttributeBool(ATTR_COLOR1_PRESENT, true) ;
            afero->setAttributeBool(ATTR_COLOR1_ENABLE, true) ;
            afero->getAttribute(ATTR_COLOR1_PWM_R) ;
            afero->getAttribute(ATTR_COLOR1_PWM_G) ;
            afero->getAttribute(ATTR_COLOR1_PWM_B) ;
        } else {
            afero->setAttributeBool(ATTR_COLOR1_PRESENT, false) ;
        }
        break ;
    case 3: /* temp */
        if (lm75b0) { /* temp with color1 */
            afero->setAttributeBool(ATTR_TEMP0_PRESENT, true) ;
            afero->setAttributeBool(ATTR_TEMP0_ENABLE, true) ;
        } else {
            afero->setAttributeBool(ATTR_TEMP0_PRESENT, false) ;
        }
        afero->setAttributeBool(ATTR_TEMP1_PRESENT, true) ; /* before */
        afero->setAttributeBool(ATTR_TEMP2_PRESENT, true) ; /* after */

        if (lm75b1) { /* temp with color2 Note: Obsolete */
            afero->setAttributeBool(ATTR_TEMP3_PRESENT, true) ;
            afero->setAttributeBool(ATTR_TEMP3_ENABLE, true) ;
        } else {
            afero->setAttributeBool(ATTR_TEMP3_PRESENT, false) ;
        }

        break ;
    case 4: /* pressure */
        if (sensor[sensor_index]) {
            afero->setAttributeBool(ATTR_GAS_PRESENT, true) ;
            afero->setAttributeBool(ATTR_GAS_ENABLE, true) ;
            afero->getAttribute(ATTR_GAS_THR_MODE) ;
            afero->getAttribute(ATTR_GAS_THR_HIGH) ;
            afero->getAttribute(ATTR_GAS_THR_LOW) ;
        } else {
            afero->setAttributeBool(ATTR_GAS_PRESENT, false) ;
        }
        break ;
    default:
        break ;
    }
    sensor_index++ ;
    return(sensor_index < NUM_MAX_SENSOR) ;
}

void edge_loop(uint32_t count_robin)
{
    static int sensor_index = 0 ;
    if ((count_robin % accel_interval) == 0) {
        if (accel) {
            accel->accum() ; /* get and accum accel data */
        }
    }

    if ((count_robin % loop_interval) == 0) {
        loop_interval = 1 ;
        if ((sensor[sensor_index])&&(sensor[sensor_index]->isEnabled())) {
            sensor[sensor_index]->runStateMachine() ;
        }
        sensor_index = (sensor_index + 1) % NUM_MAX_SENSOR ;
    }
}

int is_present(I2C *i2c, int address)
{
    char t[1] = { 0 } ;
    char data[2] = { 0, 0 } ;
    int result ;
    address <<= 1 ;
    result = i2c->write(address, t, 1, true) ;
    if (result == 0) {
        result = i2c->read(address, data, 2) ;
    }
    return((result == 0)) ;    
}

void init_sensors(void)
{
#if 0
    I2C             *edge_i2c0 = 0 ;
    I2C             *edge_i2c1 = 0 ;
    MMA8451Q        *mma8451q = 0 ;
    VEML6040        *veml6040[2] = { 0, 0 } ;
    LM75B           *lm75b0 = 0 ; /* for temp1 */
    AnalogIn        *an0 = 0 ; /* for temp2 */
    SMTC502AT       *smtc502at0 = 0 ;
    AnalogIn        *an1 = 0 ; /* for temp3 */
    SMTC502AT       *smtc502at1 = 0 ;
    LM75B           *lm75b1 = 0 ; /* for temp4 */
    AnalogIn        *an2 = 0 ; /* for gas pressure */
    PSE530          *pse530 = 0 ;
#endif

    printf("=== Initializing Sensor(s) ===\n") ;    
    edge_i2c0 = new I2C(PIN_I2C0_SDA, PIN_I2C0_SCL) ;
    edge_i2c1 = new I2C(PIN_I2C1_SDA, PIN_I2C1_SCL) ;
    
    if (is_present(edge_i2c0, SO1602A_I2C_ADDRESS)) {
        printf("SO1602A on I2C0 is present\n") ;
        display = new SO1602A(edge_i2c0, SO1602A_I2C_ADDRESS) ;
    } else if (is_present(edge_i2c1, SO1602A_I2C_ADDRESS)) {
        printf("SO1602A on I2C1 is present\n") ;
        display = new SO1602A(edge_i2c1, SO1602A_I2C_ADDRESS) ;
    } else {
        printf("SO1602A is absent\n") ;
    }
    if (display) {
        display->cls() ;
        display->locate(3, 0) ;
        display->putStr("Suntory") ;
        display->locate(0, 1) ;
        display->putStr("Server Monitor") ;
    }

    if (is_present(edge_i2c1, MMA8451Q_I2C_ADDRESS)) {
        printf("MMA8451Q on I2C1 is present\n") ;
        mma8451q = new MMA8451Q(edge_i2c1, MMA8451Q_I2C_ADDRESS) ;
        accel    = new edge_accel(mma8451q) ;
        sensor[0] = accel ;
        num_sensor++ ;
    } else {
        sensor[0] = 0 ;
        printf("MMA8451Q is absent\n") ;
    }
    
    if (is_present(edge_i2c1, VEML6040_I2C_ADDRESS)) {
        printf("VEML6040 on I2C1 is present\n") ;    
        veml6040[0] = new VEML6040(edge_i2c1, VEML6040_I2C_ADDRESS) ;
        led[0] = new PwmOut(PIN_LED_R) ; 
        led[1] = new PwmOut(PIN_LED_G) ; 
        led[2] = new PwmOut(PIN_LED_B) ; 
        color[0] = new edge_color(veml6040[0], led, pwm) ;
        sensor[1] = color[0] ;
        num_sensor++ ;
    } else {
        sensor[1] = 0 ;
        printf("VEML6040 on I2C1 is absent\n") ;
    }
    
    if (is_present(edge_i2c0, VEML6040_I2C_ADDRESS)) {
        printf("VEML6040 on I2C0 is present\n") ;    
        veml6040[1] = new VEML6040(edge_i2c0, VEML6040_I2C_ADDRESS) ;
        if (led[0] == 0) {
            led[0] = new PwmOut(PIN_LED_R) ;
            led[1] = new PwmOut(PIN_LED_G) ;
            led[2] = new PwmOut(PIN_LED_B) ;
        }
        color[1] = new edge_color(veml6040[1], led, pwm) ;
        sensor[2] = color[1] ;
        num_sensor++ ;
    } else {
        sensor[2] = 0 ;
        printf("VEML6040 on I2C0 is absent\n") ;
    }
    
    if (is_present(edge_i2c1, LM75B_I2C_ADDRESS)) {
        printf("LM75B on I2C1 is present\n") ;
        lm75b0 = new LM75B(edge_i2c1, LM75B_I2C_ADDRESS) ;
    } else {
        printf("LM75B on I2C1 is absent\n") ;
    }
#if 0    
    if (is_present(edge_i2c0, LM75B_I2C_ADDRESS)) {
        printf("LM75B on I2C0 is present\n") ;
        lm75b1 = new LM75B(edge_i2c0, LM75B_I2C_ADDRESS) ;
    } else {
        printf("LM75B on I2C0 is absent\n") ;
    }
#endif    
    an0        = new AnalogIn(PIN_AN0) ;
    smtc502at0 = new SMTC502AT(an0) ;
    an1        = new AnalogIn(PIN_AN1) ;
    smtc502at1 = new SMTC502AT(an1) ;
    temp       = new edge_temp(lm75b0, smtc502at0, smtc502at1, lm75b1) ;
    sensor[3]  = temp ;
    num_sensor++ ;
    
    an2        = new AnalogIn(PIN_AN2) ;
    pse530     = new PSE530(an2) ;
    pressure   = new edge_pressure(pse530) ;
    sensor[4]  = pressure ;
    num_sensor++ ;
 
 #if 0   
    if (color[0]) {
       color[0]->calibrate(target, color0_pwm, 10) ;
    }
    if (color[1]) {
       color[1]->calibrate(target, color1_pwm, 10) ;
    }
#endif
    if (num_sensor > 0) {
        printf("%d edge_sensor(s) registered\n", num_sensor) ;
        printf("Edge is waiting for ASR to link\n") ;
        if (display) {
            display->cls() ;
            display->locate(0, 0) ;
            display->putStr("Waiting for") ;
            display->locate(0, 1) ;
            display->putStr("ASR to Link") ;
        }
    }
}

void enable_sensors(void) 
{
    int i ;
    for (i = 0 ; i < NUM_MAX_SENSOR ; i++ ) {
        if (sensor[i]) {
            sensor[i]->enable() ;
        }
    }
}