#include "mbed.h"
#include "edge_sensor.h"
#include "edge_time.h"
#include "PSE530.h"
#include "edge_pressure.h"
#include "SO1602A.h"

extern SO1602A *display ; /* OLED display on I2C */

#define LOW_THR   0.2
#define HIGH_THR  0.3 
#define MIN_TEMP 12.0
#define MAX_TEMP 30.0

/**
 * SMC PSE530 pressure sensor
 * analog output 1.0V - 5.0V
 * 1.0V : 0
 * 5.0V : 1MPa
 * (at 0.6V : -0.1MPa)
 * Our sensor I/F converts 0-5V to 0-1
 * So we suppose V = Analog Float Value : Pressure
 * 0.2 =  0
 * 1.0 = 1MPa
 */
 
float temp2expected(float temp)
{
    const float coef_A = 0.089 ;
    const float coef_B = 0.831 ;
    float pressure ;

    pressure = temp * coef_A + coef_B ;
    return( pressure ) ;
}

edge_pressure::edge_pressure(PSE530 *pse)
{
    _pse = pse ;
    _value = 0.0 ;
    _interval = 30 ;
    _thr_mode = 0 ;
    _thr_high = HIGH_THR ;
    _thr_low = LOW_THR ;
}

edge_pressure::~edge_pressure(void)
{
    if (_pse) {
        delete _pse ;
    }
    _value = 0.0 ;
}

float edge_pressure::get_value(void)
{
    float value = 0.0 ;
    value = _pse->getPressure() ;
    return( value ) ;
}

void edge_pressure::reset(void) 
{
    _value = 0.0 ;
    _sampled_time = 0 ;
}

void edge_pressure::prepare(void) 
{
}

void edge_pressure::sample(void) 
{
    _value = get_value() ;
    _sampled_time = edge_time ;
}

void edge_pressure::set_thr_high(int16_t thr_high)
{
    switch(_thr_mode) {
    case 0: /* absolute value */
        _thr_high = (float)thr_high/100.0 ;
        break ;
    case 1: /* persentage */
        _thr_high = (float)(thr_high/100.0) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        _thr_high = (float)thr_high/100.0 ;
        break ;
    }
printf("thr_high = %.3f\n", _thr_high) ;
}

float edge_pressure::get_thr_high(float expected)
{
    float thr_high ;
    
    switch(_thr_mode) {
    case 0: /* absolute value */
        thr_high = expected + _thr_high ;
        break ;
    case 1: /* persentage */
        thr_high = expected * (1.0 + _thr_high) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        thr_high = expected + _thr_high ; /* use this as default */
        break ;
    }
    return (thr_high) ;
}

void edge_pressure::set_thr_low(int16_t thr_low)
{   
    switch(_thr_mode) {
    case 0: /* absolute value */
        _thr_low = (float)thr_low/100.0 ;
        break ;
    case 1: /* persentage */
        _thr_low = (float)(thr_low/100.0) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        _thr_low = (float)thr_low/100.0 ;
        break ;
    }
printf("thr_low = %.3f\n", _thr_low) ;
}

float edge_pressure::get_thr_low(float expected)
{
    float thr_low ;
    
    switch(_thr_mode) {
    case 0: /* absolute value */
        thr_low = expected - _thr_low ;
        break ;
    case 1: /* persentage */
        thr_low = expected * (1.0 - _thr_low) ;
        break ;
    default:
        printf("Unknown Threshold mode %d\n", _thr_mode) ;
        thr_low = expected + _thr_low ; /* use this as default */
        break ;
    }
    return (thr_low) ;
}

int edge_pressure::deliver(void) 
{
    char str_buf[32] ;
    char timestr[16] ;
    int result ;
    float expected, higher, lower ;
    print_time(_sampled_time) ;
    if (current_temp != 0) {
        sprintf(str_buf, "GAS:%.3f@%.1fC", _value, *current_temp ) ;
    } else {
        sprintf(str_buf, "GAS:%.3f ", _value  ) ;
    }
    printf(" ") ;
    printf(str_buf) ;
    if (display != 0) {
        display->cls() ;
        display->locate(0, 0) ;
        display->putStr(str_buf) ;
    }
    if (current_temp != 0) {
        expected = temp2expected(*current_temp) ;
        higher = get_thr_high(expected) ;
        lower = get_thr_low(expected) ;
        printf(" (%.3f, %.3f) ", higher, lower) ;
        if (_value > higher) {
            sprintf(str_buf, "High [exp %.2f]", expected) ;
        } else if (_value < lower) {
            sprintf(str_buf, "Low [exp %.2f]", expected) ;
        } else {
            sprintf(str_buf, "Normal") ;
        }
    }
    if (display != 0) {
        display->locate(0, 1) ;
        display->putStr(str_buf) ;
    }
    printf(" %s\n", str_buf) ;
    time2seq(_sampled_time, timestr) ;
//    printf(str_buf) ;
//    printf("\n") ;
    sprintf(_str_buf,
       "{\"DEVICE\":\"PRESS\",\"PN\":\"PSE530\",\"VAL\":\"%.3f\",\"UNIT\":\"kgf/cm2\",\"S\":\"%s\",\"E\":\"%d\"}",
       _value, timestr, _error_count) ;
    result = afero->setAttribute(1, _str_buf) ;
    return( result == afSUCCESS ) ;
}

void edge_pressure::recv_config(void)
{
}

void edge_pressure::send_config(void)
{
}