#include "MCUResetReason.hpp"

/**
  * System Reset Status Register 0 (RCM_SRS0) 0x4007_F000
  *
  * bit[7] : POR         Power-On Reset
  * bit[6] : PIN         External Reset Pin
  * bit[5] : WDOG        Watchdog
  * bit[4] : (Reserved)
  * bit[3] : LOL         Loss-of-Lock Reset
  * bit[2] : LOC         Loss-of-Clock Reset
  * bit[1] : LVD         Low-Voltage Detect Reset
  * bit[0] : WAKEUP      Low Leakage Wakeup Reset
  */
#define REG_RCM_SRS0    (uint8_t *)0x4007F000
#define POR_RESET_BIT   0x80
#define PIN_RESET_BIT   0x40
#define WDG_RESET_BIT   0x20
#define LOL_RESET_BIT   0x08
#define LOC_RESET_BIT   0x04
#define LVD_RESET_BIT   0x02
#define WUP_RESET_BIT   0x01
  /**
   * System Reset Status Register 1 (RCM_SRS1) 0x4007_F001
   *
   * bit[7:6] (Reserved)
   * bit[5] : SACKERR     Stop Mode Acknowledge Error Reset
   * bit[4] : (Reserved)
   * bit[3] : MDM_AP      MDM-AP System Reset Request
   * bit[2] : SW          Software Reset
   * bit[1] : LOCKUP      Core Lockup
   * bit[0] : (Reserved)
   */
#define REG_RCM_SRS1     (uint8_t *)0x4007F001
#define SACK_RESET_BIT   0x20
#define MDM_RESET_BIT    0x08
#define SW_RESET_BIT     0x04
#define LOCKUP_RESET_BIT 0x02
 
/**
 * Software Reset
 * 
 * From Cortex-M0 Devices Generic User Guide
 * 4.3.4 Application Interrupt and Reset Control Register
 *
 * Bit[31:16] : VECTCKEY
 * Bit[15]    : ENDIANESS
 * Bit[14:3]  : (Reserved)
 * Bit[2]     : SYSRESETREQ
 * Bit[1]     : VECTCLRACTIVE (reserved for debug use)
 * Bit[0]     : (Reserved)
 *
 * Note: To trigger software reset, both VECTKEY=0x05FA and SYSRESETREQ
 * must be written at once, therefore the value will be
 * 0x05FA0004
 */

MCUResetReason *self = NULL;

const char *MCUResetReason::str_reset_reason[] = {
    "Power On Reset",
    "External Pin Reset",
    "Watch Dog Reset : Forgot to feed?",
    "Loss of Lock Reset",
    "Loss of Clock Reset",
    "Low-Voltage Detect Reset",
    "Low Leakage Wakeup Reset",
    "Stop Mode Acknowledge Error Reset",
    "MDM-AP System Reset Request",
    "Software Reset",
    "Core Lockup Reset"
};

MCUResetReason::MCUResetReason()
{
    SRS[0] = *REG_RCM_SRS0 ;
    SRS[1] = *REG_RCM_SRS1 ;
}

MCUResetReason* MCUResetReason::ref()
{
    return self != NULL ? self : self = new MCUResetReason();
}

void MCUResetReason::clearFlag()
{
    SRS[0] = PIN_RESET_BIT;
    SRS[1] = 0;
}

MCUResetReason::RESET_REASON MCUResetReason::getResetReason(void)
{
    RESET_REASON reason;
       
    if (SRS[0] & POR_RESET_BIT) {
        reason = POWER_ON;
    } else if (SRS[0] & PIN_RESET_BIT) {
        reason = EXTERNAL_PIN;
    } else if (SRS[0] & WDG_RESET_BIT) {
        reason = WATCHDOG;
    } else if (SRS[0] & LOL_RESET_BIT) {
        reason = LOSS_OF_LOCK;
    } else if (SRS[0] & LOC_RESET_BIT) {
        reason = LOSS_OF_CLOCK;
    } else if (SRS[0] & LVD_RESET_BIT) {
        reason = LOW_VOLTAGE_DETECT;
    } else if (SRS[0] & WUP_RESET_BIT) {
        reason = LOW_LEAKAGE_WAKEUP;
    } else if (SRS[1] & SACK_RESET_BIT) {
        reason = STOP_MODE_ACK_ERROR;
    } else if (SRS[1] & MDM_RESET_BIT) {
        reason = MDM_AP_SYSTEM_RESET_REQUEST;
    } else if (SRS[1] & SW_RESET_BIT) {
        reason = SOFTWARE;
    } else if (SRS[1] & LOCKUP_RESET_BIT) {
        reason = CORE_LOCKUP;
    }
    return reason;
}

const char* MCUResetReason::getResetReasonStr()
{
    return str_reset_reason[getResetReason()];
}
 