#ifndef _PREFERENCES_HPP_
#define _PREFERENCES_HPP_

#include "mbed.h"

typedef union
{
    bool b;
    int16_t i_16;
    uint16_t ui_16;
    struct
    {
        short X, Y, Z;
    };  
    struct
    {
        uint16_t R, G, B;
    };
} VAR;

template<typename A> struct SENSORS_BASE
{
    enum INDEX
    {
        ACCELEROMETER = 0,
        COLOR,
        PRESSURE,
        CURRENT_TRANS,
        TEMPERATURE,
        SIZE,
    };
    static bool ACTIVE[SIZE];
    static bool DELTA[SIZE];
};

struct SENSORS : public SENSORS_BASE<SENSORS>
{
    template<typename A> struct COMMONPARAMS;
    struct MAX44008;
    struct VEML6040;
    struct MMA8451Q;
    template<typename A> struct SEMITEC_502AT_11_CALIB;
    template<typename A> struct DEV_TEMP;
    struct DEV_TEMP_INST;
    struct SEMITEC_502AT_11;
    struct SR_3702_150N_14Z;
    template<typename A> struct KEYENCE_PRESSURE_COMMON_PARAMS;
    struct AP_53A_KEYENCE; //AP-53A KEYENCE gas
    struct GP_M010_KEYENCE; //"GP-M010 KEYENCE" liquid
    struct NXP_LM75B;
};

template<typename A> struct SENSORS::COMMONPARAMS
{
    static char* TYPE;
    static char* PN;
    static char* UNIT;
    static char* JSON_FMT;
};

template<typename A> struct SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS
{
    static float V_REF;
    static float PRESSURE_INTERVAL;
    static float SHUNT_R;
    static float LOWEST_CUR;
    static float PRESSURE_COEF;
};

template<typename A> float SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<A>::V_REF = 3.3;
template<typename A> float SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<A>::PRESSURE_INTERVAL = 1.0;
template<typename A> float SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<A>::SHUNT_R = 165;
template<typename A> float SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<A>::LOWEST_CUR = 0.004;
template<typename A> float SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<A>::PRESSURE_COEF = 62500;

struct SENSORS::MAX44008 : public SENSORS::COMMONPARAMS<SENSORS::MAX44008>
{
    enum
    {
        ADDRESS = 0x41,
        AMB_CONFIG = 0x00, // most sensitive gain
        RAW_MODE = 0x20, // MODE_CLEAR_RGB_IR
        PWM_LED_B = 0x96
    };
    struct TRIM
    {
        enum
        {
            R = 0x50,
            G = 0x01,
            B = 0x20
        };
    };
};

template<> char* SENSORS::COMMONPARAMS<SENSORS::MAX44008>::TYPE = "COLOR";
template<> char* SENSORS::COMMONPARAMS<SENSORS::MAX44008>::PN = "MAX44008";
template<> char* SENSORS::COMMONPARAMS<SENSORS::MAX44008>::UNIT = "mW/cm2";
template<> char* SENSORS::COMMONPARAMS<SENSORS::MAX44008>::JSON_FMT(
    "{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"DATA\":[{\"TIME\":\"%ld\",\"VAL\":\"%d\",\"UNIT\":\"%s\"}]}"
);


struct SENSORS::VEML6040 : public SENSORS::COMMONPARAMS<SENSORS::VEML6040>
{
    enum
    {
        ADDRESS = 0x10,
        AMB_CONFIG = 0x00, // most sensitive gain
        RAW_MODE = 0x20, // MODE_CLEAR_RGB_IR
        PWM_LED_B = 0xFA,
        PWM_LED_R_UI16 = 0x5FA2,
        PWM_LED_G_UI16 = 0xB09B,
        PWM_LED_B_UI16 = 0x83EF
    };
    struct TRIM
    {
        enum
        {
            R = 0x50,
            G = 0x01,
            B = 0x20
        };
    };
};

template<> char* SENSORS::COMMONPARAMS<SENSORS::VEML6040>::JSON_FMT(
    "{\"DEVICE\":\"COLOR\",\"PN\":\"VEML6040\",\"VAL_R\":\"%d\",\"VAL_G\":\"%d\",\"VAL_B\":\"%d\",\"UNIT\":\"mW/cm2\",\"S\":\"%d\",\"E\":\"%d\"}"
);

struct SENSORS::MMA8451Q : public SENSORS::COMMONPARAMS<SENSORS::MMA8451Q>
{
    enum
    {
#if defined (TARGET_KL25Z)
        ADDRESS = 0x1D
#elif defined (TARGET_TEENSY3_1)
        ADDRESS = 0x1C
#endif
    };
};

template<> char* SENSORS::COMMONPARAMS<SENSORS::MMA8451Q>::JSON_FMT(
//    "{\"DEVICE\":\"ACCEL\",\"PN\":\"MMA8451Q\",\"VAL_X\":\"%.3f\",\"VAL_Y\":\"%.3f\",\"VAL_Z\":\"%.3f\",\"UNIT\":\"g\",\"N\":\"%d\",\"RET\":\"%d\"}"
    "{\"DEVICE\":\"ACCEL\",\"PN\":\"MMA8451Q\",\"VAL_X\":\"%.3f\",\"VAL_Y\":\"0\",\"VAL_Z\":\"0\",\"UNIT\":\"g\",\"S\":\"%d\",\"E\":\"%d\"}"
);

template<typename A> struct SENSORS::SEMITEC_502AT_11_CALIB
{
    static float B;
    static float T0;
    static float R0;
    static float R1;
};
template<> float SENSORS::SEMITEC_502AT_11_CALIB<SENSORS::SEMITEC_502AT_11>::B = 3324;
template<> float SENSORS::SEMITEC_502AT_11_CALIB<SENSORS::SEMITEC_502AT_11>::T0 = 298.15;
template<> float SENSORS::SEMITEC_502AT_11_CALIB<SENSORS::SEMITEC_502AT_11>::R0 = 5.0;  //kOhm
template<> float SENSORS::SEMITEC_502AT_11_CALIB<SENSORS::SEMITEC_502AT_11>::R1 = 4.95; //kOhm

template<typename A> struct SENSORS::DEV_TEMP
{
    enum
    {
        TUBE = 0,
        BEFORE_COOLING,
        AFTER_COOLING,
        SIZE
    };
    static char* JSON_FMT;
    static char* SUFFIXES[SIZE];
    static char* PART_NUMBERS[SIZE];
    static char* POSITION[SIZE];
    static char* STATUS[SIZE];
};

struct SENSORS::DEV_TEMP_INST :
    public SENSORS::DEV_TEMP<SENSORS::DEV_TEMP_INST>
{
};

template<> char* SENSORS::DEV_TEMP<SENSORS::DEV_TEMP_INST>::SUFFIXES[SIZE] = {
    "01", "02", "03"
};

template<> char* SENSORS::DEV_TEMP<SENSORS::DEV_TEMP_INST>::POSITION[SIZE] = {
    "NOT COOLED", "NOT COOLED", "BEING COOLED"
};

template<> char* SENSORS::DEV_TEMP<SENSORS::DEV_TEMP_INST>::STATUS[SIZE] = {
    "TUBE", "IN", "IN"
};

template<> char* SENSORS::DEV_TEMP<SENSORS::DEV_TEMP_INST>::PART_NUMBERS[SIZE] = {
    "LM75B", "SEMITEC 502AT-11", "SEMITEC 502AT-11"
};

template<> char* SENSORS::DEV_TEMP<SENSORS::DEV_TEMP_INST>::JSON_FMT(
//  "{\"DEVICE\":\"TEMP%s\",\"PN\":\"%s\",\"VAL\":\"%0.1f\",\"UNIT\":\"degC\"}"
//    "{\"DEVICE\":\"TEMP%s\",\"PN\":\"%s\",\"POS\":\"%s\",\"STAT\":\"%s\",\"VAL\":\"%0.1f\",\"UNIT\":\"degC\"}"
    "{\"DEVICE\":\"TEMP%s\",\"PN\":\"%s\",\"POS\":\"%s\",\"STAT\":\"%s\",\"VAL\":\"%0.1f\",\"UNIT\":\"degC\",\"S\":\"%d\",\"E\":\"%d\"}"
);

struct SENSORS::SEMITEC_502AT_11 :
    public SENSORS::COMMONPARAMS<SENSORS::SEMITEC_502AT_11>,
    public SENSORS::SEMITEC_502AT_11_CALIB<SENSORS::SEMITEC_502AT_11>
{
};

//template<> char* SENSORS::COMMONPARAMS<SENSORS::SEMITEC_502AT_11>::TYPE = "TEMP";
template<> char* SENSORS::COMMONPARAMS<SENSORS::SEMITEC_502AT_11>::PN = "SEMITEC 502AT-11";
//template<> char* SENSORS::COMMONPARAMS<SENSORS::SEMITEC_502AT_11>::UNIT = "degC";

//template<> char* SENSORS::COMMONPARAMS<SENSORS::SEMITEC_502AT_11>::JSON_FMT(
//    "{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"DATA\":[{\"TIME\":\"%ld\",\"VAL\":\"%0.1f\",\"UNIT\":\"%s\",\"POS\":\"%s\"}]}"
//);

struct SENSORS::NXP_LM75B :
    public SENSORS::COMMONPARAMS<SENSORS::NXP_LM75B>
{
    enum
    {
        ADDRESS   = 0x48,
        REG_Conf  = 0x01,
        REG_Temp  = 0x00,
        REG_Tos   = 0x03,
        REG_Thyst = 0x02
    };
};

template<> char* SENSORS::COMMONPARAMS<SENSORS::NXP_LM75B>::PN = "LM75B";

struct SENSORS::SR_3702_150N_14Z : public SENSORS::COMMONPARAMS<SENSORS::SR_3702_150N_14Z>{};

template<> char* SENSORS::COMMONPARAMS<SENSORS::SR_3702_150N_14Z>::TYPE = "CURRENT_TRANS";;
template<> char* SENSORS::COMMONPARAMS<SENSORS::SR_3702_150N_14Z>::PN = "SR-3702-150N/14Z";
template<> char* SENSORS::COMMONPARAMS<SENSORS::SR_3702_150N_14Z>::UNIT = "V";
template<> char* SENSORS::COMMONPARAMS<SENSORS::SR_3702_150N_14Z>::JSON_FMT(
    "{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"DATA\":[{\"TIME\":\"%ld\",\"VAL\":\"%0.1f\",\"UNIT\":\"%s\"}]}"
);
struct SENSORS::AP_53A_KEYENCE :
    public SENSORS::COMMONPARAMS<SENSORS::AP_53A_KEYENCE>,
    public SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<SENSORS::AP_53A_KEYENCE> {};
template<> char *SENSORS::COMMONPARAMS<SENSORS::AP_53A_KEYENCE>::TYPE = "GAS PRESSURE";
template<> char *SENSORS::COMMONPARAMS<SENSORS::AP_53A_KEYENCE>::PN = "AP-53A KEYENCE";
template<> char *SENSORS::COMMONPARAMS<SENSORS::AP_53A_KEYENCE>::UNIT = "KPa";
template<> char* SENSORS::COMMONPARAMS<SENSORS::AP_53A_KEYENCE>::JSON_FMT(
    //"{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"DATA\":[{\"TIME\":\"%ld\",\"VAL\":\"%0.1f\",\"UNIT\":\"%s\"}]}"
    "{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"VAL\":\"%0.1f\"}"
);

struct SENSORS::GP_M010_KEYENCE :
    public SENSORS::COMMONPARAMS<SENSORS::GP_M010_KEYENCE>,
    public SENSORS::KEYENCE_PRESSURE_COMMON_PARAMS<SENSORS::GP_M010_KEYENCE> {};
template<> char *SENSORS::COMMONPARAMS<SENSORS::GP_M010_KEYENCE>::TYPE = "LIQUID PRESSURE";
template<> char *SENSORS::COMMONPARAMS<SENSORS::GP_M010_KEYENCE>::PN = "AGP-M010 KEYENCE";
template<> char *SENSORS::COMMONPARAMS<SENSORS::GP_M010_KEYENCE>::UNIT = "KPa";
template<> char* SENSORS::COMMONPARAMS<SENSORS::GP_M010_KEYENCE>::JSON_FMT(
    //"{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"DATA\":[{\"TIME\":\"%ld\",\"VAL\":\"%0.1f\",\"UNIT\":\"%s\"}]}"
    "{\"DEVICE\":\"%s\",\"PN\":\"%s\",\"VAL\":\"%0.1f\"}"
);

struct PINS
{
    struct LED_4_COLOR;
    struct COLOR;
    struct ACCELEROMETER;
    struct CURRENT_TRANS;
    struct TEMPERATURE;
    struct UART;
    struct ASR_1;
    struct GAS;
    struct LIQUID;
};

struct PINS::LED_4_COLOR
{
    enum
    {
        R = PTA5,
        G = PTA4,
        B = PTA12
    };
};
    
struct PINS::COLOR
{
    enum
    {
        SDA = PTE0,
        SCL = PTE1
    };
};

struct PINS::ACCELEROMETER
{
    enum
    {
#if defined (TARGET_KL25Z)
        SDA = PTE25,
        SCL = PTE24
#elif defined (TARGET_TEENSY3_1)
        SDA = PTB3,
        SCL = PTB2
#endif
    };
};

struct PINS::CURRENT_TRANS
{
    enum
    {
        AIN = PTB0
    };
};

struct PINS::TEMPERATURE
{
    enum
    {
        BEFORE_COOLING = PTB0, // la suno shield
//      BEFORE_COOLING = PTC2, // Hibiki only
        AFTER_COOLING = PTB1,
        SDA = PTE0,
        SCL = PTE1
    };
};

struct PINS::GAS
{
    enum
    {
        AIN = PTB2
    };
};

struct PINS::LIQUID
{
    enum
    {
        AIN = PTB3
    };
};

struct PINS::ASR_1
{
    struct SPI;
    enum
    {
#if defined (TARGET_KL25Z)
        RESET = PTC9 //PSOC
        //RESET = PTA20 
#elif defined (TARGET_TEENSY3_1)
        RESET = PTD6
#endif
    };
    struct SIG
    {
        template<typename A> struct RESET_BASE
        {
            static time_t INTERVAL;            
        };
        struct RESET : public RESET_BASE<RESET>
        {
            enum
            {
                ASSERT = 0,
                DEASSERT = 1
            };
        };
    };
};

template<> time_t PINS::ASR_1::SIG::RESET_BASE<PINS::ASR_1::SIG::RESET>::INTERVAL = 60*5;

struct PINS::ASR_1::SPI
{
    enum
    {
#if defined (TARGET_KL25Z)
        MOSI = PTD2,
        MISO = PTD3,
        SCK  = PTD1,
        CS   = PTD0,
        SR   = PTD4 // request from slave
#elif defined (TARGET_TEENSY3_1)
        MOSI = PTC6,
        MISO = PTC7,
        SCK  = PTC5,
        CS   = PTC4,
        SR   = PTD1 // request from slave
#endif
    };
    struct SIG
    {
        struct CS
        {
            enum 
            {
                ASSERT = 0,
                DEASSERT = 1
            };
        };
    };
    enum
    {
        NUM_BITS_PER_FRAME = 8,
        MODE_0 = 0,
        FREQUENCY = 1*1000*1000
    };
};

struct PINS::UART
{
    enum
    {
        BAUD_RATE = 115200
    };
#if defined (TARGET_TEENSY3_1)
    enum
    {
        uart0_tx = PTA2,
        uart0_rx = PTA1
    };
#endif
#if defined (TARGET_KL25Z) && (WICED_SMART_UART_CONNECTION)
    struct WICED
    {
        struct TAG3
        {
            enum
            {
                TO_RX = PTE22 // -> TAG3 J7-1
                ,TO_TX = PTE23 // -> TAG3 J8-5
                //power 3v3 -> TAG J9-1
                //GND -> TAGJ9-4
            };
        };
    };
#endif
};

template<typename A> struct PREFERENCES_BASE
{
    static int32_t SENSING_INTERVAL[SENSORS::SIZE];
    static time_t EPOCH_UTC;
    typedef union
    {
        uint32_t ui32;
        uint8_t  ui8[sizeof(uint32_t)];
    } _crc32;
    static _crc32 CRC32;
    static bool FLOW_CONTROL;
    static bool DBG_PRINT_ENABLED;
    static bool DBG_PRINT_ASR_ENABLED;
    struct AFERO_ATTRIBUTE
    {
        uint32_t ATTR_CRC32;
    };
};

struct PREFERENCES : PREFERENCES_BASE<PREFERENCES>
{
    struct NOTIFICATION;
    struct DATA_TRANSFER_MODE
    {
        enum
        {
            JSON = 0,
            BASE64 = 1
        };
    };
    struct THRESHOLD
    {
        struct RANGE
        {
            enum
            {
                MIN = 0,
                MAX,
                DELTA,
                SIZE
            };
        };
    };
    template<typename A> struct NOTIFICATION_BASE
    {
        static bool ENABLED[SENSORS::SIZE][THRESHOLD::RANGE::SIZE];
        static VAR THRESHOLD[SENSORS::SIZE][THRESHOLD::RANGE::SIZE]; // not used ... for now.
    };
};

template<> int32_t PREFERENCES_BASE<PREFERENCES>::SENSING_INTERVAL[SENSORS::SIZE] = { 30, 30, 30, 30, 30 };
template<> time_t PREFERENCES_BASE<PREFERENCES>::EPOCH_UTC = 0;
template<> PREFERENCES_BASE<PREFERENCES>::_crc32 PREFERENCES_BASE<PREFERENCES>::CRC32 = { 0 };

template<> bool PREFERENCES_BASE<PREFERENCES>::FLOW_CONTROL = false;

template<> bool PREFERENCES_BASE<PREFERENCES>::DBG_PRINT_ENABLED = true;
template<> bool PREFERENCES_BASE<PREFERENCES>::DBG_PRINT_ASR_ENABLED = true;

struct PREFERENCES::NOTIFICATION : public PREFERENCES::NOTIFICATION_BASE<PREFERENCES::NOTIFICATION>
{
};

template<> bool PREFERENCES::NOTIFICATION_BASE<PREFERENCES::NOTIFICATION>::ENABLED[SENSORS::SIZE][THRESHOLD::RANGE::SIZE];
template<> VAR  PREFERENCES::NOTIFICATION_BASE<PREFERENCES::NOTIFICATION>::THRESHOLD[SENSORS::SIZE][THRESHOLD::RANGE::SIZE];

typedef union
{
    typedef struct
    {
        time_t timeStamp;
        SENSORS::INDEX index;
    } PACKET_BASE;
    struct : PACKET_BASE
    {
        short x,y,z;
    } accelerometer;
    struct : PACKET_BASE
    {
        uint16_t R, G, B;
    } color;
    struct : PACKET_BASE
    {
        float val;
    } temperature;
    struct : PACKET_BASE
    {
        float val;
    } current_trans;
    struct : PACKET_BASE
    {
        float val;
    } pressure;
} PACKET;

//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, false, false, false, true }; // temperature
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, true, false, false, false }; // color
template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { true, true, false, false, true }; // color + temp + accelerometer
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { true, false, false, false, true }; // temp + accelerometer
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, true, false, false, true }; // color + temp
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, true, false, true, true }; // sensor box
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, false, true, false, false }; // pressure
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { true, false, false, false, false }; // accelerometer
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, true, false, false, false }; // color only ... sensor box
//template<> bool SENSORS_BASE<SENSORS>::ACTIVE[SENSORS::SIZE] = { false, false, false, true, false }; // CT only ... sensor box

template<> bool SENSORS_BASE<SENSORS>::DELTA[SENSORS::SIZE] = { true, false, false, false, false }; // accelerometer

#endif //_PREFERENCES_HPP_