#include "mbed.h"

/* Spaceball Button bit-masks */
#define SBALL_BUTTON_1        0x0001   /* bit  0 */
#define SBALL_BUTTON_2        0x0002   /* bit  1 */
#define SBALL_BUTTON_3        0x0004   /* bit  2 */
#define SBALL_BUTTON_4        0x0008   /* bit  3 */
#define SBALL_BUTTON_5        0x0010   /* bit  4 */
#define SBALL_BUTTON_6        0x0020   /* bit  5 */
#define SBALL_BUTTON_7        0x0040   /* bit  6 */
#define SBALL_BUTTON_8        0x0080   /* bit  7 */
#define SBALL_BUTTON_9        0x0100   /* bit  8 */
#define SBALL_BUTTON_10       0x0200   /* bit  9 */
#define SBALL_BUTTON_11       0x0400   /* bit 10 */
#define SBALL_BUTTON_12       0x0800   /* bit 11 */

/* The Spaceball 3003 and 3003 FLX only have "left" and "right" buttons */
#define SBALL_BUTTON_RIGHT    0x1000   /* bit 12 */
#define SBALL_BUTTON_LEFT     0x2000   /* bit 13 */

/* The Spaceball 2003A and 2003B have a dedicated pick button on the ball */
/* The Spaceball 2003 FLX uses "button 9" as the pick button.             */
/* All of them return this as "button 9" in their encoded button data     */
#define SBALL_BUTTON_PICK      SBALL_BUTTON_RIGHT   /* bit  12 */

/* On Spaceball 2003A and 2003B, the Rezero is "button 8" on the device */
/* On the newer devices, there are dedicated rezero buttons */
#define SBALL_BUTTON_REZERO  0x4000   /* bit 14 */

/* The Spaceball 4000 FLX has a configurable palm rest which can be in    */
/* either "left" or "right" handed mode.  When it is configured in "left" */
/* handed mode, the "lefty" bit is set, and coordinate systems need to be */
/* inverted on one axis.                                                  */
#define SBALL_MODE_LEFTY     0x8000   /* bit 15 */


enum eSpaceBallAxis { 
    TX=0, TY, TZ, 
    Right=0, Forward, Up, 
    RX=3, RY, RZ,
    Pitch=3, Roll, Yaw };

class SpaceBall {
public:
    
    SpaceBall ( PinName tx, PinName rx, bool bSpaceOrb=false );
    ~SpaceBall() {};

    void Init();
    
    void GetTranslation ( float* fValue[3] ) const
    {
        *fValue[0] = GetAxis ( TX );
        *fValue[1] = GetAxis ( TY );
        *fValue[2] = GetAxis ( TZ );
    }
    void GetRotation ( float* fValue[3] ) const
    {
        *fValue[0] = GetAxis ( RX );
        *fValue[1] = GetAxis ( RY );
        *fValue[2] = GetAxis ( RZ );
    }
    float GetAxis ( int nAxis ) const
    {
        return GetAxisRaw ( nAxis ) * 0.00005f * _fScale;
    }
    int GetAxisRaw ( int nAxis ) const
    {
        if ( nAxis<0 || nAxis>5 )
            return 0;
        
        return m_axis[nAxis];
    }
    float operator[] (eSpaceBallAxis nAxis) const
    {   return GetAxis ( nAxis ); }
    float operator[] (int nAxis) const
    {   return GetAxis ( nAxis ); }
    
    int GetButtons() const
    {
        return m_buttons;
    }
    operator int (void) const
    {   return GetButtons(); }
    
    void SetScale ( float fScale=1.0f )
    {   _fScale = fScale; }
    float GetScale ( void )
    {   return _fScale; }


protected:
    Serial _serial;
    bool _bSpaceOrb;
    float _fScale;
    
    int m_axis[6];      /* last translational data received */
    //int m_rot[3];        /* last rotational data received */
    int m_buttons;       /* current button status */
    
    void SerialISR(void);
    void Process ( char c );
    
    virtual void DoChangedAxis (void) {};
    virtual void DoChangedButtons (void) {};
    
private:
   unsigned char m_buf[256];
    char m_resetstring[256]; 
    int m_bufpos;        /* current char position in packet buffer */
    int m_packtype;      /* what kind of packet is it */
    int m_packlen;       /* how many bytes do we ultimately expect? */
    int m_escapedchar;   /* if set, we're processing an escape sequence */
    int m_erroroccured;  /* if set, we've received an error packet or packets */
    int m_resetoccured;  /* if set, ball was reset, so have to reinitialize it */
    int m_spaceball4000; /* if set, its a Spaceball 4000 */
    int m_leftymode4000; /* if set, Spaceball 4000 in "lefty" orientation */
    int m_timer;         /* time since last packet was received */

    void InitSB() {};
    void InitSO() {};

    void ProcessSB ( char c );
    void ProcessSO ( char c );
};