/*
* @author Jochen Krapf
* parts by Andy Kirkham
*
* @section LICENSE
*
* Copyright (c) 2014 Jochen Krapf, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without restriction,
* including without limitation the rights to use, copy, modify, merge, publish, distribute,
* sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*
* @section DESCRIPTION
* mbed QEI (quadrature encoder interface) library, for decoding AB signals from a rotary encoder
*
* Use cases:
* - Rotary encoder in closed loop motor regulation
* - Hand wheel
* - Input device for motion control (MoCo)
*/

#include "mbed.h"
#include "FPointer_vi.h"

/** Quadrature Encoder Interface for Motion Control.
*
* A class to decode pulses on a rotary encoder with AB signals (quadrature encoder). 
* It uses all 4 edges of the AB signals to increase the counter resolution 4 times of cycles per rotation/revolution (CPR) (e.g. an encoder with 500 CPR get 2000 counts per rotation)
*
* In opposite to most common QEI implementation this is resistant to jitter and chatter on AB signals and motor vibrations. 
* When using interrupts (IRQ_NO_JAMMING-mode) only the needed edge and pin is activated to prevent jamming CPU time with unnecessary interrupts.
* Whes reaching the next position the edge that triggerd this position (state) is ignored to aboid oscillating up/down counts.
*
* It can also be used in polling mode i.g. in idle routines if interrupts are not desired. 
* At this mode be sure that the sampling frequency is heigher than the maximum rotation speed (expeced counts per second)
*
* The internal state machine is based on a look up table (LUT) to minimize interrupt retention time and get all necessary flags at once.
*
* Additional the rotation speed of the encoder can be measured. The algorithm is based on the measuring time between the edges to get a very precise speed at very slow rotation.
*
* The library is designed to support closed loop speed- and motion-controller for also slow and smooth motions like movie camera motion control.
*
* Example:
*
* @code
#include "mbed.h"
#include "QEIx4.h"

DigitalOut LEDalive(LED1);
DigitalOut LEDzero(LED2);
DigitalOut LEDup(LED4);
DigitalOut LEDdown(LED3);

Timer t;   // timer for polling

// ports for nxp LPC 1768
QEIx4 qei1(p30, p29, p28, (QEIx4::EMODE)(QEIx4::IRQ | QEIx4::SPEED));   // QEI with index signal for zeroing
QEIx4 qei2(p21, p22, NC,  QEIx4::IRQ_NO_JAMMING);                       // QEI with AB signals only
QEIx4 qei3(p25, p24, NC,  QEIx4::POLLING);                              // QEI without interrups in polling mode

// The callback functions
void myCounterChangeCallback(int value)
{
    static int valueLast=-1;

    if ( value > valueLast ) {
        LEDup = !LEDup;
        LEDdown = 0;
    } else {
        LEDdown = !LEDdown;
        LEDup = 0;
    }
    valueLast = value;
}

void myIndexTriggerCallback(int value)
{
    qei1 = 0;   // reset counter
    LEDzero = 1;
}

int main()
{
    t.start();

    qei1.setIndexTrigger(true);     // set the flag to zero counter on next index signal rises
    qei1.setSpeedFactor(1.0f);      // factor to scale from Hz (edges pe second = 4 * CPS) to user units (1.0=Hz, 1/(4*CPR)=rps, 1/(60*4*CPR)=rpm, 360/(4*CPR)=°/s, ...)
    qei3.attachIndexTrigger(myIndexTriggerCallback);
    
    qei3.attachCounterChange(myCounterChangeCallback);

    while(1) {
        qei3.poll();   // poll manually without interrupt - sampling in this loop with about 2kHz

        if ( t.read_ms() > 250 ) { // every quater second (4 Hz)
            t.reset();
            t.start();
            LEDalive = !LEDalive;

            printf ( "\r\n%6d  %6d  %6d  %10.3f", (int)qei1, (int)qei2, (int)qei3, (float)qei1.getSpeed() );   // print counter values
        }

        wait_us(20);   // for about 50kHz polling
    }
}
* @endcode
*/
class QEIx4
{
public:

    typedef enum EMODE {
        POLLING = 0,
        IRQ = 1,
        IRQ_NO_JAMMING = 2,
        SPEED = 4,
    } EMODE;

    /** constructor of QEIx4 object
    *
    * @param pinA     Pin number of input/interrupt pin for encoder line A. All port pins are possible except p19 and p20
    * @param pinB     Pin number of input/interrupt pin for encoder line B. All port pins are possible except p19 and p20
    * @param pinI     Pin number of input pin for optional encoder index or reference switch.
    * @param eMode    Flag to use interrups to detect changes on line A and B. For none interrupt use mode POLLING and call the function poll() frequently. For optional speed calculation the mode SPEED can be ored
    */
    QEIx4 ( PinName pinA, PinName pinB, PinName pinI=NC, EMODE eMode=IRQ );

    /** destructor of QEIx4 object
    */
    ~QEIx4();

    /** Gets the actual counter value.
    *
    * @return        Actual counter value
    */
    int read() {
        return _counter;
    }

    /** Gets the actual counter value as int operator.
    *
    * @return        Actual counter value as int operator
    */
    operator int () {   // int-Operator
        return _counter;
    }

    /** Sets the counter value at actual encoder position to given value.
    *
    * @param        Counter value
    */
    void write ( int counter ) {
        _counter = counter;
    }

    /** Sets the counter value at actual encoder position to given value as assign operator.
    *
    * @param        Counter value
    */
    int operator= ( int counter ) {   // Assign-Operator
        write(counter);
        return counter;
    }

    /** Polls the state machine manually and updates the counter value.
    */
    void poll () {
        ProcessISR();
    }

    /** Sets the flag for zeroing on next high on index pin while AB lines triggers next counting. The trigger calls tha callback function in which the counter can be set to zero or the actual counter can be latched in for later offset calculation
    *
    * @param        Flag for triggering. Set to 1 for call the attached callback. It is reseted after this call
    */
    void setIndexTrigger ( bool bIndexTrigger ) {
        _bIndexTrigger = bIndexTrigger;
    }

    /** attach - Overloaded attachment function.
    *
    * Attach a C type function pointer as the callback.
    *
    * Note, the callback function prototype must be:-
    * @code
    * void myCallbackFunction(int);
    * @endcode
    * @param A C function pointer to call.
    */
    void attachCounterChange(void (*function)(int) = 0) {
        fPointerCounterChange.attach (function);
    }

    /** attachCounterChange - Overloaded attachment function.
     *
     * Attach a C++ type object/method pointer as the callback.
     *
     * Note, the callback method prototype must be:-
     * @code
     *     public:
     *         static void myCallbackFunction(int);
     * @endcode
     * @param A C++ object pointer.
     * @param A C++ method within the object to call.
     */
    template<class T>
    void attachCounterChange(T* item, void (T::*method)(int)) {
        fPointerCounterChange.attach( item, method);
    }

    /** attachDirectionChange - Overloaded attachment function.
    *
    * Attach a C type function pointer as the callback.
    *
    * Note, the callback function prototype must be:-
    * @code
    * void myCallbackFunction(int);
    * @endcode
    * @param A C function pointer to call.
    */
    void attachDirectionChange(void (*function)(int) = 0) {
        fPointerDirectionChange.attach (function);
    }

    /** attachDirectionChange - Overloaded attachment function.
     *
     * Attach a C++ type object/method pointer as the callback.
     *
     * Note, the callback method prototype must be:-
     * @code
     *     public:
     *         static void myCallbackFunction(int);
     * @endcode
     * @param A C++ object pointer.
     * @param A C++ method within the object to call.
     */
    template<class T>
    void attachDirectionChange(T* item, void (T::*method)(int)) {
        fPointerDirectionChange.attach( item, method);
    }

    /** attachIndexTrigger - Overloaded attachment function.
    *
    * Attach a C type function pointer as the callback.
    *
    * Note, the callback function prototype must be:-
    * @code
    * void myCallbackFunction(int);
    * @endcode
    * @param A C function pointer to call.
    */
    void attachIndexTrigger(void (*function)(int) = 0) {
        fPointerIndexTrigger.attach (function);
    }

    /** attachIndexTrigger - Overloaded attachment function.
     *
     * Attach a C++ type object/method pointer as the callback.
     *
     * Note, the callback method prototype must be:-
     * @code
     *     public:
     *         static void myCallbackFunction(int);
     * @endcode
     * @param A C++ object pointer.
     * @param A C++ method within the object to call.
     */
    template<class T>
    void attachIndexTrigger(T* item, void (T::*method)(int)) {
        fPointerIndexTrigger.attach( item, method);
    }

    /** Sets the factor for the getter-functions to convert in another unit (1.0=Hz, 1/(4*CPR)=rps, 1/(60*4*CPR)=rpm, 360/(4*CPR)=°/s, ...)
    *
    * @param fSpeedFactor - factor to scale from Hz (edges per second = 4 * CPS) to user units
    */
    void setSpeedFactor(float fSpeedFactor) {
        _fSpeedFactor = fSpeedFactor;
    }

    /** Gets the actual speed as float value. The value is scales by the facor set by setPositionFactor()
    *
    * @return        Actual encoder speed as float
    */
    float getSpeed();

    /** Sets the factor for the getter-functions to convert in another unit (e.g. CPR (cycles per rotation) * 4.0 to get 1.0 for a full rotation)
    *
    * @param fPositionFactor  Factor to scale from counts to user unit
    */
    void setPositionFactor ( float fPositionFactor ) {
        _fPositionFactor = fPositionFactor;
    }

    /** Gets the actual counter value as float value. The value is scales by the facor set by setSpeedFactor()
    *
    * @return        Actual encoder position as float
    */
    float getPosition () {
        return (float)_counter * _fPositionFactor;
    }


protected:
    InterruptIn _pinA, _pinB;
    DigitalIn _pinI;
    FPointer_vi fPointerCounterChange;
    FPointer_vi fPointerDirectionChange;
    FPointer_vi fPointerIndexTrigger;

    int _counter;
    short _state;
    short _eMode;
    bool _bIndexTrigger;

    Timer _SpeedTimer;

    unsigned int _nSpeedLastTimer;
    unsigned int _nSpeedTimeoutMax;
    unsigned int _nSpeedTimeoutCount;
    int _nSpeedAvrTimeSum;
    int _nSpeedAvrTimeCount;
    float _fLastSpeed;

    void ProcessISR ( void );
    void callback_timeout();

    float _fPositionFactor;
    float _fSpeedFactor;

private:
    static short _modeLUT[32];
};