// Version BSW 1.0   erstellt am 18.08.2015
// 

#include "mbed.h"
#include "main.h"
#include "timer0.h"
#include "monitor.h"
#include "MODSERIAL.h"

// timer 2 => optischer Alarm Periode = 500 entspricht Blinkfrequnz von 1 Selunde
#define OA_PERIODE 500
// timer 3 => Minimale Zeit für den akustischen Alarm in ms >> soll 10000 entspricht 10 Sekunden
#define AA_NIN 10000
// timer 4 => Verzögerung für des Entriegeln des Tasters. In dieser Zeit wird dann kein neuer
// akustischer Alarm ausgelöst
#define AA_DELAY 10000

#define NEED_CONSOLE_OUTPUT 1   // Set this if you need debug messages on the console
                                // mit Debug wird die Programmgröße von 32,8k um ca. 300 Byte vergrößert
                                // dies ist nicht bedeutend, aber auch die Ausführungszeit der Ausgabebefehle
                                // benötigt CPU Zeit
#if NEED_CONSOLE_OUTPUT
  #define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
  #define DEBUG(...)    // nothing
#endif                  // #if NEED_CONSOLE_OUTPUT 

//------------------------------------------------------------------------------
// Anlegen von Klassen

timer0 down_timer;                              // Zeitsteuerung
MODSERIAL pc(USBTX, USBRX);                     // tx, rx

monitor mon;

Timer t;                                        // Zeitmessung für Entprellen

//------------------------------------------------------------------------------
// Zuordnung von Eingängen und Ausgängen

DigitalOut LED(PTE30);

DigitalOut OUT1(PTC6);      // Summer
DigitalOut OUT2(PTC5);      // LED Rot
DigitalOut OUT3(PTC4);      // LED Grün
DigitalOut OUT4(PTC3);      // LED Gelb
DigitalOut OUT5(PTC2);      // nicht belegt
DigitalOut OUT6(PTC1);      // Relais

DigitalIn IN1(PTC7);        // Notaus, schaltet gegen 24V =>            24V oder 1 ist der Ruhezustand
DigitalIn IN2(PTD4);        // Sabotageschalter, schaltet gegen 24V =>  24V oder 1 ist Ruhezustand
DigitalIn IN3(PTD5);        // KABA Türöffner Signal =>                 24V oder 1 ist Ruheezustand
DigitalIn IN4(PTD6);        // nicht belegt                             
DigitalIn IN5(PTD7);        // nicht belegt 

DigitalInOut rst(PTA4);     //Connect this to the reset pin

//------------------------------------------------------------------------------
// Globale Variablen

bool in;

struct di_struct in1;      // Eingang 1
struct di_struct in2;      // Eingang 2
struct di_struct in3;      // Eingang 3

//------------------------------------------------------------------------------
//

int main()
{
    rst.input();     // SW watchdog ist derzeit nicht erforderlich
    
    //------------------------------------------------------------------------------
    // RS232 Schnittstellt welche auf den CMSIS-DAP (USB Port) weitergeleitet wird
    //
    pc.baud(115200);
    pc.printf("\n N1380 V1.0 was compiled on %s  %s \n", __DATE__,__TIME__);


    //--------------------------------------------------------------------
    //  Anfangswerte setzen

    in1.old = 1;        // 1 im Ruhezustand
    in1.aktiv = 0;      
    in1.filter = 0;
    in1.optischer_alarm = 0;
    in1.summer = 0;
    
    in2.old = 1;        // 1 im Ruhezustand
    in2.aktiv = 0;      
    in2.filter = 0;
    in2.optischer_alarm = 0;
    in2.summer = 0;

    in3.old = 1;        // 1 im Ruhezustand
    in3.aktiv = 0;      
    in3.filter = 0;
    
    //--------------------------------------------------------------------
    // Softwaretimer für die Zeitsteuerung anlegen

    down_timer.SetCountdownTimer(0,1,500);      // 1 ms Timer * 500 => ergibt 500 ms
    down_timer.SetCountdownTimer(1,1,100);      // 1 ms Timer * 10 => ergibt 10 ms, 100 ms für Einschaltverzögerung
    down_timer.SetCountdownTimer(2,1,0);        // 1 ms Timer => reserviert für optischen Alarm
    down_timer.SetCountdownTimer(3,1,0);        // 1 ms Timer => reserviert für akustischen Alarm
    down_timer.SetCountdownTimer(4,1,0);        // 1 ms Timer => reserviert für nach Entriegelung
    
    //--------------------------------------------------------------------
    // Schleife fuer die Datenerfassung

    while(1) 
    {
        //-------------------------------------------
        // Prüfen ob Zeichen eingegeben wurden
        // wird in der aktiven Version ausgeblendet, ist nur zum Testen
        
        mon.get_line();

        //-------------------------------------------
        // timer 0 steuert die LED auf dem Board mit der Takrate 0,5 Sekunden

        if (down_timer.GetTimerStatus(0) == 0) {
            down_timer.SetCountdownTimer(0,1,500);
            LED = !LED;
            
        }

        //-------------------------------------------
        // Eingänge abfragen und Aktionen ableiten
        // down

        if (down_timer.GetTimerStatus(1) == 0) 
        {
            //down_timer 1 mit 10 ms gesetzt
            down_timer.SetCountdownTimer(1,1,10);

            //------------------------------------------------
            // IN1  Nottaster auswerten, null im Ruhezustand
            // wird mit der negativen Flanke aktiviert
            // das Signal muss mindestens 10ms anliegen, damit es akzeptiert wird
        
            in = IN1;
            
            
            if(!in && in1.old)       // Öffner im Ruhezustand 24 V >> durch Drücken wird eine neg. Flanke erzeugt
            {
                in1.old = 0;
                in1.aktiv = 1;       // Taster ist betätigt
                in1.optischer_alarm = 1;
                in1.summer = 1;
                down_timer.SetCountdownTimer(3,1,AA_NIN);  // Zeit für Softtimer auf 10 Sekunden setzen
                DEBUG("\n negative Flanke IN1 \n");
            }

            if (in && !in1.old) {     // Öffner im Ruhenzustand 24 V >> durch Rückstellen enteht eine pos. Flanke
                in1.old = 1;
                in1.aktiv = 0;        // Taster ist zurückgesetzt
                
                in1.summer = 0;
                down_timer.SetCountdownTimer(4,1,AA_DELAY);  // Zeit für Entriegelung, Haube innerhalb von 10 Sekunden schließen
                DEBUG("\n positve Flanke IN1 \n");
            }
            
            // optischer Alarm ausschalten, wenn IN3 wieder 0 bekommt ==> Quittierung

            if (in1.optischer_alarm && in3.aktiv && !in1.aktiv) 
            {
                in1.optischer_alarm = 0;
                DEBUG("\n IN1 mit IN3 quittiert \n");
            }


            //------------------------------------------------
            // IN2 Sabotageschalter, 24V ist Ruhezustand
            // wird mit der negativen Flanke aktiviert
            // das Signal muss mindestens 10ms anliegen, damit es akzeptiert wird
            
            in = IN2;
            
            
            if(!in && in2.old)          // Eingang neg. Flanke
            {    
                in2.old = 0;            // 0 im aktivierten Modus
                in2.aktiv = 1;          // Eingang ist aktiv
                //in2.optischer_alarm = 1;
                in2.summer = 1;      // am 11.05.15 entfernt, da Summer weiter untern aktiviert wird
                //down_timer.SetCountdownTimer(3,1,AA_NIN);  // Zeit für Softtimer auf 10 Sekunden setzen
                DEBUG("\n negative Flanke IN2 \n");
            }

            if (in && !in2.old)         // Eingang mit pos. Flanke
            {    
                in2.old = 1;            // 1 im Ruhezustand   
                in2.aktiv = 0;          // Eingang ist inaktiv
                in2.summer = 0;
                DEBUG("\n positve Flanke IN2 \n");
            }
/*
            if (in2.aktiv && !in2.summer && !in1.summer && (down_timer.GetTimerStatus(4) == 0))
            {
                in2.summer = 1; 
                down_timer.SetCountdownTimer(3,1,AA_NIN);  // Zeit für Softtimer auf 10 Sekunden setzen ( zum Testen auf 2 Sek. gesetzt )  
                DEBUG("\n IN2 Summer aktiviert \n");
            }
            
            // nur zurückschalten, wenn IN3 wieder 1 ==> Quittierung

            if (in2.optischer_alarm && in3.aktiv) {
                in2.optischer_alarm = 0;
                DEBUG("\n IN2 mit IN3 quittiert \n");
            }
*/
            //------------------------------------------------
            // IN3 Signalstation, Öffner => 1 ist Ruheezustand

            in = IN3;

            if(in && !in3.old)        // Eingang pos. Flanke
            {   
                in3.old = 1;
                in3.aktiv = 0;        // Eingang ist Ruhezustand
                
            }

            if (!in && in3.old)       // Eingang mit neg Flanke
            {
                in3.old = 0;
                in3.aktiv = 1;        // Eingang ist aktiv    
            }


            //---------------------------------------------------------------------------------------------------------------------------------------
            // Die Eingänge sind nun gelesen, jetzt kommt das Steuern der Ausgaänge
            //
            
            //------------------------------------------------
            // Rot/Grüne LED (verriegelt/entriegelt):
            // rot      Tor ist verriegelt
            // grün     Tor ist entriegelt
            
            //  2. IN3 auf 0 = OUT2 aus und OUT3 ein   Rot aus / Grün an   (entriegelt)
            //  3. IN1 auf 1 = OUT2 aus und OUT3 ein   Rot aus / Grün an   (entriegelt)
            
            if (in3.aktiv || in1.aktiv) {          
                OUT2 = LED_AUS;
                OUT3 = LED_EIN;
            }
            else 
            //  1. IN3 auf 1 = OUT2 ein und OUT3 aus   Rot an / Grün aus   (verriegelt)
            {   
                OUT2 = LED_EIN;
                OUT3 = LED_AUS;
            }            
            
            //------------------------------------------------
            //  Gelbe LED (optischer Alarm):

            if (in1.optischer_alarm)  // sobald der optische Alarm aus Eingang 1 akiviert wurde, diesen in Blinkmode schalten
            {    
                if (down_timer.GetTimerStatus(2) == 0) 
                {
                    down_timer.SetCountdownTimer(2,1,OA_PERIODE);  // Zeit für Softtimer auf eine Sekunde setzen
                    OUT4 = !OUT4;
                }
            } 
            else 
            {
                OUT4 = LED_AUS;
            }

            //------------------------------------------------
            // Summer (OUT1 GND)


            if (in1.summer || in2.summer || (down_timer.GetTimerStatus(3) != 0))  // sobald der Eingang 1 oder Eingang 2 akiviert ist oder der Timer
            {                                                                     // noch aktiv ist, den Summer aktivieren
               DEBUG("\n Summer 1 = %d Summer 2 = %d \n",(uint8_t)in1.summer,(uint8_t)in2.summer);
               OUT1 = 1;
            } 
            else 
            {
                OUT1 = 0;
            }
            
            //------------------------------------------------
            // Relais K1:
            // Der Kontakt des Nottasters wird weitergegeben

            if (in1.aktiv) {  // 1.  IN1 auf 0 = OUT6 (K1) ein
                OUT6 = 1;
            } else {          // 2.  IN1 auf 1 = OUT6 aus
                OUT6 = 0;
            }

        } // end if (downtimer ...
        
    } // end while

} // end main

