// Copyright 2013 Pervasive Displays, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at:
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
// express or implied.  See the License for the specific language
// governing permissions and limitations under the License.

#ifndef __EPD_H__
#define __EPD_H__

#include <SPI.h>
#include <mbed.h>

typedef enum {
    EPD_1_44,        // 128 x 96
    EPD_2_0,         // 200 x 96
    EPD_2_7          // 264 x 176
} EPD_size;

typedef enum {           // Image pixel -> Display pixel
    EPD_compensate,  // B -> W, W -> B (Current Image)
    EPD_white,       // B -> N, W -> W (Current Image)
    EPD_inverse,     // B -> N, W -> B (New Image)
    EPD_normal       // B -> B, W -> W (New Image)
} EPD_stage;

typedef void EPD_reader(void *buffer, uint32_t address, uint16_t length);

class EPD_Class
{
    public:
    uint16_t lines_per_display;
    
    private:

    DigitalOut EPD_Pin_EPD_CS;
    DigitalOut EPD_Pin_PANEL_ON;
    DigitalOut EPD_Pin_BORDER;
    DigitalOut EPD_Pin_DISCHARGE;
    PwmOut EPD_Pin_PWM;
    DigitalOut EPD_Pin_RESET;
    DigitalIn EPD_Pin_BUSY;
    
    EPD_size size;
    uint16_t stage_time;

    uint16_t factored_stage_time;
    uint16_t dots_per_line;
    uint16_t bytes_per_line;
    uint16_t bytes_per_scan;
    const uint8_t *gate_source;
    uint16_t gate_source_length;
    const uint8_t *channel_select;
    uint16_t channel_select_length;

    bool filler;

    public:
    unsigned char lineDta[33];

    public:
    EPD_Class(PinName Pin_EPD_CS, PinName Pin_PANEL_ON, PinName Pin_BORDER, PinName Pin_DISCHARGE, PinName Pin_PWM, PinName Pin_RESET, PinName Pin_BUSY);
    // power up and power down the EPD panel
    void begin(EPD_size sz);
    void start();
    void end();

    void setFactor(int temperature = 25) {
        this->factored_stage_time = this->stage_time * this->temperature_to_factor_10x(temperature) / 10;
    }

    // clear display (anything -> white)
    void clear(int from_line = 0, int to_line = -1)
    {
        if(to_line == -1){
            to_line = this->lines_per_display;
        }
        this->frame_fixed_repeat(0xff, EPD_compensate, from_line, to_line);
        this->frame_fixed_repeat(0xff, EPD_white, from_line, to_line);
        this->frame_fixed_repeat(0xaa, EPD_inverse, from_line, to_line);
        this->frame_fixed_repeat(0xaa, EPD_normal, from_line, to_line);
    }

    // assuming a clear (white) screen output an image
    void image(const uint8_t *image, int from_line = 0, int to_line = -1)
    {
        if(to_line == -1){
            to_line = this->lines_per_display;
        }
        this->frame_fixed_repeat(0xaa, EPD_compensate, from_line, to_line);
        this->frame_fixed_repeat(0xaa, EPD_white, from_line, to_line);
        this->frame_data_repeat(image, EPD_inverse, from_line, to_line);
        this->frame_data_repeat(image, EPD_normal, from_line, to_line);
    }



    // Low level API calls
    // ===================

    // single frame refresh
    void frame_fixed(uint8_t fixed_value, EPD_stage stage, int from_line, int to_line);
    void frame_data(const uint8_t *new_image, EPD_stage stage, int from_line, int to_line);

    // stage_time frame refresh
    void frame_fixed_repeat(uint8_t fixed_value, EPD_stage stage, int from_line, int to_line);
    void frame_data_repeat(const uint8_t *new_image, EPD_stage stage, int from_line, int to_line);

    // convert temperature to compensation factor
    int temperature_to_factor_10x(int temperature);

    // single line display - very low-level
    void line(uint16_t line, const uint8_t *data, uint8_t fixed_value, bool read_progmem, EPD_stage stage);
};

extern EPD_Class EPD;

#endif