/**
 * Copyright 2014 Nordic Semiconductor
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License
 */

#include "mbed.h"
#include "IRSender.h"

// Remove the log level define in production. It will prevent your mbed from entering low power modes.
#define LOG_LEVEL_INFO 
#include "Log.h"
#include "Puck.h"

Puck* puck = &Puck::getPuck();

IRSender irpin(p14);

const UUID IR_SERVICE_UUID = stringToUUID("bftj ir         ");
const UUID COMMAND_UUID    = stringToUUID("bftj ir command ");
const UUID DATA_UUID       = stringToUUID("bftj ir data    ");
const UUID PERIOD_UUID     = stringToUUID("bftj ir period  ");

#define DATA_BUFFER_SIZE 200
unsigned int dataBuffer[DATA_BUFFER_SIZE];
uint8_t period = 26;
int receiveIndex;

#define COMMAND_BEGIN_CODE_TRANSMISSION 0
#define COMMAND_END_CODE_TRANSMISSION 1

void onCommandWrite(const uint8_t* value, uint8_t length) {
    switch(value[0]) {
        case COMMAND_BEGIN_CODE_TRANSMISSION:
            receiveIndex = 0;
            break;
        case COMMAND_END_CODE_TRANSMISSION:
            puck->disconnect();
            LOG_INFO("Going to fire IR code...\n");
            irpin.irSeq(period, receiveIndex, dataBuffer);
            LOG_INFO("Fire complete!\n");
            break;
    }
}


void onDataWrite(const uint8_t* value, uint8_t length) {
    for(int i = 0; i < 20 && receiveIndex < length; i += 2) {
        dataBuffer[receiveIndex++] = (value[i] << 8) | value[i + 1];
    }
}


void onPeriodWrite(const uint8_t* value, uint8_t length) {
    period = value[0];
}


int main() {
    puck->addCharacteristic(IR_SERVICE_UUID, COMMAND_UUID, 1);
    puck->addCharacteristic(IR_SERVICE_UUID, DATA_UUID, 20);
    puck->addCharacteristic(IR_SERVICE_UUID, PERIOD_UUID, 1);
    puck->init(0xABBA);

    puck->onCharacteristicWrite(&COMMAND_UUID, onCommandWrite);
    puck->onCharacteristicWrite(&DATA_UUID, onDataWrite);
    puck->onCharacteristicWrite(&PERIOD_UUID, onPeriodWrite);

    while (puck->drive());
}