#include <mbed.h>
#include "EPD.h"
#include "lz.h"

// Remove the log level define in production. It will prevent your mbed from entering low power modes.
#define LOG_LEVEL_ERROR
#include "Puck.h"

int receiveIndex = 0;

Puck* puck = &Puck::getPuck();


EPD_Class EPD(p0, p2, p3, p8, p5, p6, p7);


const UUID DISPLAY_SERVICE_UUID = stringToUUID("bftj display    ");
const UUID COMMAND_UUID = stringToUUID("bftj display com");
const UUID DATA_UUID = stringToUUID("bftj display dat");


 /*
 
 connections:

 
LPC1768 <--> nRF51822 <--> Shield
---------||------------||--------
  p12   <-->    p2    <-->   D2 (M_EPD_PANEL_ON)
  p13   <-->    p3    <-->   D3 (M_EPD_BORDER)
  p14   <-->    p4    <-->   D4 (M_/SD_CS)
  p25   <-->    p5    <-->   D5 (M_EPD_PWM)
  p16   <-->    p6    <-->   D6 (M_EPD_/RESET) 
  p17   <-->    p7    <-->   D7 (M_EPD_BUSY)
  p18   <-->    p8    <-->   D8 (M_EPD_DISCHARGE)
  p19   <-->   p26    <-->   D9 (M_/WORD_STOCK_CS)
  p20   <-->    p0    <-->   D10 (M_/EPD_CS)
  p21   <-->   p27    <-->   A1 (M_OE123)
  p22   <-->   p28    <-->   A2 (M_CKV)
  p23   <-->   p29    <-->   A3 (M_STV_IN)
   
  p5   <-->    p20    <-->   MOSI
  p6   <-->    p22    <-->   MISO
  p7   <-->    p25    <-->   SCK
  
  VCC <--> VCC
  GND <--> GND
 
 */
 
#define COMMAND_NOOP 0
#define COMMAND_CLEAR 1
#define COMMAND_IMAGE_UPPER 2
#define COMMAND_IMAGE_LOWER 3
#define COMMAND_BEGIN_UPPER 4
#define COMMAND_BEGIN_LOWER 5

#define IMAGE_SIZE 2904
#define BUFFER_SIZE 2917
#define Y_SIZE 88
#define X_SIZE 264
uint8_t buffer[BUFFER_SIZE];

int currentCommand = COMMAND_NOOP;


void reverseBufferSegment(int from, int to) {
    for(int i = from; i < (to + from + 1) / 2; i++) {
        int i2 = to + from - i - 1;
        int temp = buffer[i];
        buffer[i] = buffer[i2];
        buffer[i2] = temp;
    }
    LOG_DEBUG("Reversed buffer from index %i to index %i.\n", from, to);
}


void onCommandWritten(const uint8_t* value, uint8_t length) {
    currentCommand = value[0];
    LOG_DEBUG("Received command: %i.\n", currentCommand);
    
    switch(currentCommand) {
        
        case COMMAND_BEGIN_UPPER:
        case COMMAND_BEGIN_LOWER:
            receiveIndex = BUFFER_SIZE;
            break;
        
        case COMMAND_CLEAR:
            LOG_INFO("Clearing display.\n");
            EPD.begin(EPD_2_7);
            EPD.start();
            EPD.clear();
            EPD.end();
            break;
    
        case COMMAND_IMAGE_UPPER:
            LOG_INFO("Writing image to top half of display.\n");
            reverseBufferSegment(receiveIndex, BUFFER_SIZE);
            LOG_DEBUG("LZ_Uncompress(in: 0x%x, out 0x%x, insize: %i)\n", buffer + receiveIndex, buffer, BUFFER_SIZE - receiveIndex);
            LZ_Uncompress(buffer + receiveIndex, buffer, BUFFER_SIZE - receiveIndex);
            LOG_INFO("Uncompressed %i bytes of data.\n", BUFFER_SIZE - receiveIndex);
            EPD.begin(EPD_2_7);
            EPD.start();
            EPD.image(buffer, 0, EPD.lines_per_display / 2);
            EPD.end();
            break;
        
        case COMMAND_IMAGE_LOWER:
            LOG_INFO("Writing image to bottom half of display.\n");
            reverseBufferSegment(receiveIndex, BUFFER_SIZE);
            LZ_Uncompress(buffer + receiveIndex, buffer, BUFFER_SIZE - receiveIndex);
            LOG_INFO("Uncompressed %i bytes of data.\n", BUFFER_SIZE - receiveIndex);
            EPD.begin(EPD_2_7);
            EPD.start();
            EPD.image(buffer, EPD.lines_per_display / 2, EPD.lines_per_display);
            EPD.end();
            break;
        
    }
    
    currentCommand = COMMAND_NOOP;
}

void onDataWritten(const uint8_t* value, uint8_t length) {
    LOG_VERBOSE("Data written, receiveIndex: %i\n", receiveIndex);
    for(int i = 0; i < length && receiveIndex > 0; i++) {
        buffer[--receiveIndex] = value[i];
    }
}


int main() {
    LOG_INFO("Starting puck.\n");
    
    DigitalOut SD_CS(p4);
    DigitalOut WORD_STOCK_CS(p26);
      
    SD_CS = 1;
    WORD_STOCK_CS = 1;
    
    puck->addCharacteristic(DISPLAY_SERVICE_UUID, COMMAND_UUID, 1);
    puck->addCharacteristic(DISPLAY_SERVICE_UUID, DATA_UUID, 20);
    
    
    puck->onCharacteristicWrite(&COMMAND_UUID, onCommandWritten);
    puck->onCharacteristicWrite(&DATA_UUID, onDataWritten);
    
    puck->init(0x5EED);
    
    puck->countFreeMemory();
    
    currentCommand = COMMAND_NOOP;    

    while (puck->drive());   
}