#include "mbed.h"
#include "EthernetInterface.h"
#include "nsdl_support.h"
#include "dbg.h"
#include "GroveColourSensor.h"
#include "ColorDetector.h"
#include "color_detector.h"
#include <cmath>

Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

// I2C Settings
#define COLOR_SENSOR_SCL I2C_SCL
#define COLOR_SENSOR_SDA I2C_SDA

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "0.0.0.0"
#define MASK    "255.255.255.0"
#define GW      "0.0.0.0"

// NSP configuration
/* Change this IP address to that of your mbed Device Server installation */
static const char* NSP_ADDRESS = "api.connector.mbed.org";
static const int NSP_PORT = 5683;
char endpoint_name[] = "nespresso-machine-ethernet";
uint8_t ep_type[] = {"nespresso-endpoint"};
char nsp_domain[] = "56645321f5e24c49908e42f4d71b9ccb";
uint8_t lifetime_ptr[] = {"1200"};

// ****************************************************************************
// Ethernet initialization

EthernetInterface eth;

static void ethernet_init()
{
    // Initialize network
#ifdef DHCP
    NSDL_DEBUG("DHCP in use\r\n");
    eth.init();
#else
    eth.init(IP, MASK, GW);
#endif
    if(eth.connect(30000) == 0)
        pc.printf("Connect OK\n\r");

    NSDL_DEBUG("IP Address:%s ", eth.getIPAddress());
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    
    NSDL_DEBUG("Creating resources");

    // Create resources
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Dynamic resources
    create_color_detector_resource(resource_ptr);
    
    // Register with NSP
    register_endpoint(true);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

// ****************************************************************************
// Program entry point

int main()
{
    NSDL_DEBUG("mbed Nespresso Demo\n");
    
    // Begin Color Sensor init
    
    // First, ensure I2C bus is released by toggling clock (fixes reset errors)
    DigitalOut scl_dummy(COLOR_SENSOR_SCL);
    
    for (int i = 0; i < 100; i++) {
        scl_dummy = !scl_dummy;
        wait_us(2);
    }
    
    // Next, initialize I2C
    I2C i2c(COLOR_SENSOR_SDA, COLOR_SENSOR_SCL);
    
    // Create color sensor instance
    GroveColourSensor colorSensor(&i2c);
    
    // Various config
    colorSensor.powerUp();
    colorSensor.setGain(3);
    colorSensor.setBlockRead();
    
    // Create color detector
    ColorDetector detector(&colorSensor, A0);
    
    // Pass reference to color detector
    set_color_detector(&detector);
    
    // Initialize Ethernet interface first
    ethernet_init();
    
    // Initialize NSP node
    nsp_init();
    
    // Initialize NSDL stack
    nsdl_init();
    
    // Create NSDL resources
    create_resources();
    
    nsdl_event_loop_init();
    
    while(true) {
        // Check if any nsdl events need to be handled
        nsdl_event_loop_run_once();
        
        // Sample the color detector and potentially send samples to mDS
        run_color_detector();
    }
}
