/*
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __GROVE_COLOUR_SENSOR_H__
#define __GROVE_COLOUR_SENSOR_H__

#include "mbed.h"

/** RGBC sample struct
 *  Used for storing 16 bit values of red, green, blue, and clear light channels from the sensor.
 */
struct RGBC_Sample {
    uint16_t red, green, blue, clear;
};

/** RGBC union
 *  This union wraps the RGBC_Sample struct into an iterable data structure.
 */
union RGBC {
    RGBC_Sample ch;
    uint16_t data[4];
};

/** GroveColourSensor class.
 *  Library for using the Grove colour sensor: http://www.seeedstudio.com/depot/Grove-I2C-Color-Sensor-p-854.html
 */
class GroveColourSensor {
public:

    /** Colour enum
     *  This enum provides a standard way of enumerating the colour channels. The order they are listed is the order in which the values are read from the sensor.
     */
    enum Colour_t {
        GREEN = 0,
        RED,
        BLUE,
        CLEAR,
        NUM_COLOURS
    };        

    /**
     *   Constructor, doesn't affect the device at all.
     *
     * @param i2c A pointer to the initialized I2C instance.
     */
    GroveColourSensor(I2C *i2c);
    
    /**
     *   Powers up the colour sensor.
     *   Returns true if successful, otherwise returns false.
     */
    bool powerUp(void);
    
    /**
     *   Powers down the colour sensor.
     *   Returns true if successful, otherwise returns false.
     */
    bool powerDown(void);
    
    /**
     *   Set the gain of the colour sensor.
     *
     *   The following are valid gain values:
     *   
     *   - 0 - 1X gain
     *   - 1 - 4X gain
     *   - 2 - 16X gain
     *   - 3 - 64X gain
     *   
     * @param gain The gain value specified above.
     */
    void setGain(uint8_t gain);
    
    /**
     *   Read a specific colour.
     *   
     * @param colour The colour to read (RED, GREEN, BLUE, or CLEAR).
     */
    uint16_t readColour(Colour_t colour);
    
    /**
     *   Non enum version of readColour()
     *   
     * @param colour The integer corresponding to the colour you wish to read.
     */
    uint16_t readColour(unsigned colour);
    
    /**
     *   Configures the colour sensor to expect block reads. Call this before calling readBlock().
     */
    void setBlockRead();
    
    /**
     *   Read the red, green, blue, and clear channels in one transaction. You must call setBlockRead() before calling this function.
     *   
     * @param sample A pointer to the RGBC instance you wish to populate.
     */
    void readBlock(RGBC *sample);
    


private:
    static const uint8_t SEVEN_BIT_ADDRESS = 0x39;
    I2C *i2c;
    
    /**
     *   Helper function to convert two bytes into a uint16_t.
     *   
     * @param lowByte Least signficant byte.
     * @param highByte Most signficant byte.
     */
    uint16_t bytesTo16bit(char lowByte, char highByte);
};

#endif /* #ifndef __GROVE_COLOUR_SENSOR_H__ */
