/*
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "GroveColourSensor.h"

GroveColourSensor::GroveColourSensor(I2C *i2c) : i2c(i2c) {
    // Empty
}


bool GroveColourSensor::powerUp(void) {
    static const char powerUpCommand[] = {0x80, 0x03};
    return i2c->write((SEVEN_BIT_ADDRESS << 1), powerUpCommand, sizeof(powerUpCommand)) == 0;
}

bool GroveColourSensor::powerDown(void) {
    static const char powerDownCommand[] = {0x80, 0x00};
    return i2c->write((SEVEN_BIT_ADDRESS << 1), powerDownCommand, sizeof(powerDownCommand)) == 0;
}

void GroveColourSensor::setGain(uint8_t gain) {
    // Set gain (0 Prescale)
    char gainRegValue = 0x00 | (gain << 4);
    
    char gainCommand[] = {0x87, gainRegValue};
    i2c->write((SEVEN_BIT_ADDRESS << 1), gainCommand, sizeof(gainCommand));
}

uint16_t GroveColourSensor::readColour(Colour_t colour) {
    char readColourRegistersCommand = 0xb0 + (2 * static_cast<unsigned>(colour));
    i2c->write((SEVEN_BIT_ADDRESS << 1), &readColourRegistersCommand, 1 /* size */);

    uint16_t colourValue;
    i2c->read((SEVEN_BIT_ADDRESS << 1), reinterpret_cast<char *>(&colourValue), sizeof(uint16_t));
    return colourValue;
}

void GroveColourSensor::setBlockRead() {
    char blockReadCommand = 0xD0;
    i2c->write((SEVEN_BIT_ADDRESS << 1), &blockReadCommand, 1);
}

void GroveColourSensor::readBlock(RGBC *sample) {
    char tmpColours[8] = {0};
    
    if (!i2c->read((SEVEN_BIT_ADDRESS << 1), tmpColours, sizeof(tmpColours))) {        
        sample->ch.green = bytesTo16bit(tmpColours[0], tmpColours[1]);
        sample->ch.red = bytesTo16bit(tmpColours[2], tmpColours[3]);
        sample->ch.blue = bytesTo16bit(tmpColours[4], tmpColours[5]);
        sample->ch.clear = bytesTo16bit(tmpColours[6], tmpColours[7]);
    } else {
        printf("I2C Read error\r\n");
    }
}

uint16_t GroveColourSensor::readColour(unsigned colour) {
    if (colour >= NUM_COLOURS) {
        return 0;
    }

    return readColour(static_cast<Colour_t>(colour));
}

uint16_t GroveColourSensor::bytesTo16bit(char lowByte, char highByte) {
    uint16_t res = highByte;
    res = (res << 8) | lowByte;
    return res;
}
