/*
 * Builds the payload to send to the server.
 *
 * Copyright (c) 2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <mbed.h>
#include "payload.h"

// Maximum number of samples to include in the payload.
// Defined such that the payload fits within one UDP packet.
#define MAX_SAMPLES 300

Payload::Payload() :
    m_mdb_bytes(0)
{
}

//
// Build the payload. The payload is defined by the Payload::MeanDeltaBuffer struct 
// and consists of: 
//   * number of samples (32 bits)
//   * mean of red, green, blue channels (32 bits each)
//   * difference of sample to mean for red, green, blue channels (1 byte each)   
//
// This encoding can store ~300 samples in a one UDP packet.
//
void Payload::build(const RGBC* samples, int sample_count)
{
    if (sample_count > MAX_SAMPLES)
        sample_count = MAX_SAMPLES;
        
    for (int i = 0; i < sample_count; i++)
    {
        m_mdb.mean[0] += (samples[i].data[0]);
        m_mdb.mean[1] += (samples[i].data[1]);
        m_mdb.mean[2] += (samples[i].data[2]);
    }
    
    m_mdb.sample_count = sample_count;
    m_mdb.mean[0] /= sample_count;
    m_mdb.mean[1] /= sample_count;
    m_mdb.mean[2] /= sample_count;
    
    for (int i = 0; i < sample_count; i++)
    {
        // TODO: clamp to signed char range [-128, 127]
        m_mdb.deltas[i][0] = m_mdb.mean[0] - (samples[i].data[0]);
        m_mdb.deltas[i][1] = m_mdb.mean[1] - (samples[i].data[1]);
        m_mdb.deltas[i][2] = m_mdb.mean[2] - (samples[i].data[2]);
    }    

    // calculate number of bytes in payload    
    m_mdb_bytes = sizeof(int) + // number of samples
        (3 * sizeof(int)) + // mean of red, green & blue channels
        (sample_count * sizeof(signed char) * 3); // one byte per channel per sample
}

int Payload::raw_bytes_size() 
{ 
    return m_mdb_bytes; 
}

uint8_t* Payload::raw_bytes() const 
{ 
    return (uint8_t *)&m_mdb; 
} 
