#include "mbed.h"
//#include "EthernetInterface.h"
#include "SNIC_WifiInterface.h"
#include "nsdl_support.h"
#include "dbg.h"
#include "GroveColourSensor.h"
#include "ColorDetector.h"
#include "color_detector.h"
#include <cmath>
#include "Endpoint.h"
#include "UDPSocket.h"
#include "ExternalFlashClient.h"

Serial pc(USBTX, USBRX); // tx, rx

// ****************************************************************************
// Configuration section

#define COLOR_SENSOR

// I2C Settings
#define COLOR_SENSOR_SCL I2C_SCL
#define COLOR_SENSOR_SDA I2C_SDA

// Envoy Settings
uint32_t SSID_ADDRESS   = 0x0100;
uint32_t KEY_ADDRESS    = 0x0180;

// Pins and SPI module for the external flash class
PinName nSCS = D10;
PinName SCLK = D13;
PinName MOSI = D11;
PinName MISO = D12;
PinName SIRQ = D9;

// buffer for storing reads
uint8_t readBuffer[100];
block_t block = { .data = readBuffer,
                  .length = sizeof(readBuffer),
                  .offset = 0,
                  .maxLength = sizeof(readBuffer)
                };

typedef enum {
    STATE_SSID_READ,
    STATE_SSID_WAIT,
    STATE_KEY_READ,
    STATE_KEY_WAIT,
    STATE_WIFI,
    STATE_NSDL,
    STATE_MAIN
} state_t;

volatile static state_t state;
volatile static bool button1Pressed = false;
volatile static bool button2Pressed = false;

// debug LEDs
DigitalOut red(LED1);
DigitalOut green(LED2);
DigitalOut blue(LED3);

// Ethernet configuration
/* Define this to enable DHCP, otherwise manual address configuration is used */
#define DHCP

/* Manual IP configurations, if DHCP not defined */
#define IP      "0.0.0.0"
#define MASK    "255.255.255.0"
#define GW      "0.0.0.0"

//#define DEMO_AP_SSID                  "demossid"
#define DEMO_AP_SECURITY_TYPE         e_SEC_WPA2_AES
//#define DEMO_AP_SECUTIRY_KEY          "mbed.org"
//#define DEMO_AP_SECUTIRY_KEY_LEN      8

char SSID[50] = {0};
char KEY[50]  = {0};

// NSP configuration
/* Change this IP address to that of your mbed Device Server installation */
static const char* NSP_ADDRESS = "23.99.29.171"; // aka connector.mbed.org
static const int NSP_PORT = 5683;
char endpoint_name[] = "nespresso-machine-wifi-envoy";
uint8_t ep_type[] = {"nespresso-endpoint"};
char nsp_domain[] = "56645321f5e24c49908e42f4d71b9ccb";
uint8_t lifetime_ptr[] = {"1200"};

// ****************************************************************************
// Ethernet initialization
C_SNIC_WifiInterface  eth( D1, D0, NC, NC, D3 );

// Envoy initialization
SPI spi(MOSI, MISO, SCLK);
ExternalFlashClient flash(spi, nSCS, SIRQ);

// liveness LED
void tickerISR()
{
    // LED is off when high
    switch(state)
    {
        case STATE_SSID_READ:
        case STATE_SSID_WAIT:
        case STATE_KEY_READ:
        case STATE_KEY_WAIT:
                            red = !red;
                            break;
        
        case STATE_WIFI:
        case STATE_NSDL:
                            red = 1;
                            blue = !blue;
                            break;
                            
        case STATE_MAIN:
                            red = 1;
                            blue = 1;                            
                            green = 0;
                            break;
        default:
                            red = 0;
                            break;
    }
}

static void ethernet_init()
{
    char* ip = NULL;
    
    do {
        // Initialize Wi-Fi interface
        int s = eth.init();
    
        NSDL_DEBUG("init();\r\n");
    
        if( s != 0 ) {
            NSDL_DEBUG( "Could not initialise wifi\n\r" );
            //return -1;
        }
        wait(0.5);
        s = eth.disconnect();
        NSDL_DEBUG("disconnect();\r\n");
        if( s != 0 ) {
            NSDL_DEBUG( "disconnect failed\r\n" );
            //return -1;
        }
    
        wait(0.3);
        // Connect AP
        eth.connect( SSID
                     , strlen(SSID)
                     , DEMO_AP_SECURITY_TYPE
                     , KEY
                     , strlen(KEY));
        NSDL_DEBUG("connect();\r\n");
        wait(0.5);
    
        NSDL_DEBUG("IP Config();\r\n");
        eth.setIPConfig( true );
        
        ip = eth.getIPAddress();
    } while (ip == NULL);

    NSDL_DEBUG("IP Address:%s ", ip);
}

// ****************************************************************************
// NSP initialization

UDPSocket server;
Endpoint nsp;

static void nsp_init()
{
    server.init();
    server.bind(NSP_PORT);

    nsp.set_address(NSP_ADDRESS, NSP_PORT);

    NSDL_DEBUG("name: %s", endpoint_name);
    NSDL_DEBUG("NSP=%s - port %d\n", NSP_ADDRESS, NSP_PORT);
}

// ****************************************************************************
// Resource creation

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;

    NSDL_DEBUG("Creating resources");

    // Create resources
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr) {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Dynamic resources
    create_color_detector_resource(resource_ptr);

    // Register with NSP
    register_endpoint(true);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

//Read the Key
void readDone2()
{
    // copy string. first byte is the length
    uint8_t length = block.data[0];
    
    if (length < sizeof(KEY))
    {
        memcpy(KEY, &(block.data[1]), length);
        KEY[length] = '\0';
    
        // debug output the entire buffer
        for (int idx = 0; idx < block.length; idx++) {
            // zero out content to verify next read
            block.data[idx] = 0;
        }

        state = STATE_WIFI;    
    }
}

// Read the SSID
void readDone()
{
    // copy string. first byte is the length
    uint8_t length = block.data[0];
    
    if (length < sizeof(SSID))
    {
        memcpy(SSID, &(block.data[1]), length);
        SSID[length] = '\0';
    
        // debug output the entire buffer
        for (int idx = 0; idx < block.length; idx++) {
            // zero out content to verify next read
            block.data[idx] = 0;
        }

        state = STATE_KEY_READ;
    }
}

// interrupt from flash signaling new data is available
// TODO: implement log on flash so clients can see whether
// they should read out more values
void flashUpdated()
{
    // update state. read flash is called from main, not interrupt context
    if (state == STATE_SSID_WAIT)
    {
        state = STATE_SSID_READ;
    }
    else if (state == STATE_MAIN)
    {
        // restart system when new credentials have been entered
        NVIC_SystemReset();
    }
}

void button1ISR()
{
    button1Pressed = true;
}

void button2ISR()
{
    button2Pressed = true;
}

// ****************************************************************************
// Program entry point

int main()
{
    // blink LEDs
    Ticker ticker;
    ticker.attach(tickerISR, 1.0);
    
    // leds are off when high
    red = 1;
    green = 1;
    blue = 1;
        
    // setup buttons
    InterruptIn button1(PTC6);
    button1.mode(PullUp);
    // Delay for initial pullup to take effect
    wait(.01);
    button1.fall(button1ISR);

    InterruptIn button2(PTA4);
    button2.mode(PullUp);
    // Delay for initial pullup to take effect
    wait(.01);
    button2.fall(button2ISR);    
    
    NSDL_DEBUG("mbed Nespresso Demo\n");

    // register flash update callback
    flash.onUpdates(flashUpdated);

#ifdef COLOR_SENSOR
    // Begin Color Sensor init

    // First, ensure I2C bus is released by toggling clock (fixes reset errors)
    DigitalOut scl_dummy(COLOR_SENSOR_SCL);

    for (int i = 0; i < 100; i++) {
        scl_dummy = !scl_dummy;
        wait_us(2);
    }

    // Next, initialize I2C
    I2C i2c(COLOR_SENSOR_SDA, COLOR_SENSOR_SCL);

    // Create color sensor instance
    GroveColourSensor colorSensor(&i2c);

    // Various config
    colorSensor.powerUp();
    colorSensor.setGain(3);
    colorSensor.setBlockRead();

    // Create color detector
    ColorDetector detector(&colorSensor, A0);

    // Pass reference to color detector
    set_color_detector(&detector);
#endif

    state = STATE_SSID_READ;

    for (;;)
    {
        // high priority
        if (button1Pressed)
        {
            button1Pressed = false;
            printf("button 1\r\n");
            flash.erase();
        }
        else if (button2Pressed)
        {
            button2Pressed = false;
            printf("button 2\r\n");
        }
        else
        {
            switch(state)
            {
                case STATE_SSID_READ:
                                printf("\r\n");
                                printf("read SSID from flash\r\n");
                                printf("waiting for SSID");
                                fflush(stdout);
                                flash.read(SSID_ADDRESS, &block, readDone);
                                
                                state = STATE_SSID_WAIT;
                                break;
                                
                case STATE_SSID_WAIT:            
                                wait(1); // TODO: look into timing issues on the NRF51
                                printf(".");
                                fflush(stdout);
                                break;
    
                case STATE_KEY_READ:
                                printf("read KEY from flash\r\n");
                                flash.read(KEY_ADDRESS, &block, readDone2);
                                
                                state = STATE_KEY_WAIT;
                                break;
        
                case STATE_KEY_WAIT:
                                break;
                                
                case STATE_WIFI:
                                printf("ssid: %s, key: %s\r\n", SSID,KEY);
    
                                // Initialize Ethernet interface first
                                ethernet_init();
                                
                                state = STATE_NSDL;
                                break;
                                
                case STATE_NSDL:
                                // Initialize NSP node
                                nsp_init();
                            
                                // Initialize NSDL stack
                                nsdl_init();
                            
                                // Create NSDL resources
                                create_resources();
                            
                                nsdl_event_loop_init();
                                
                                state = STATE_MAIN;
                                break;
    
                case STATE_MAIN:
                                // Check if any nsdl events need to be handled
                                nsdl_event_loop_run_once();
                        
#ifdef COLOR_SENSOR
                                // Sample the color detector and potentially send samples to mDS
                                run_color_detector();
#else
                                wait(1);
                                printf(".");
                                fflush(stdout);
#endif
                                
                                break;
                default:
                                printf("unknown state\r\n");
                                break;
            }
        }
    }
}
