#ifndef _COLOR_DETECTOR_H_
#define _COLOR_DETECTOR_H_

#include "mbed.h"
#include "GroveColourSensor.h"

#define CD_BUFF_LEN 512
#define CD_PR_BUFF_LEN 30


/** ColorDetector class.
 *  Used for detecting changes in color readings from a Grove colour sensor: http://www.seeedstudio.com/depot/Grove-I2C-Color-Sensor-p-854.html.
 *  Also uses a photoresistor to detect when an object passes between the colour sensor and the photoresistor.
 */
class ColorDetector {
public:
    /**
     *    Constructor.
     *
     *    @param sensor Pointer to instantiated and powered up Grove colour sensor.
     *    @param photoresistorPin The pin of the photoresistor.
     *    @param prThreshold The value that the photoresistor must dip below before samples are taken.
     *        NOTE: Lower this if no samples are being taken.
     *    @param minNumSamples The minimum number of samples that must be taken for a capture event to be considered valid.
     *        NOTE: Lower this if no samples are being taken.
     */
    ColorDetector(GroveColourSensor *sensor, PinName photoresistorPin, float prThreshold = 0.55, int minNumSamples = 100);
    
    /**
     *    Takes a new reading from the sensor. If the value of the photoresistor is below prThreshold,
     *    then the color sensor is sampled. Once the photoresistor value rises above prThresholdagain,
     *    it returns the number of samples it took. Otherwise, it returns 0;
     */
    int sample();
    
    /**
     *    Returns a pointer to the sample buffer (pointer does not change).
     */
    RGBC* getBuffer();
    
private:
    GroveColourSensor *sensor;
    AnalogIn photoresistor;
    float prThreshold;
    int minNumSamples;
    RGBC* curBufPtr;
    bool capturing;
    RGBC sample_buf[CD_BUFF_LEN];
};
    
#endif